# Copyright (C) 2025 Intel Corporation
# SPDX-License-Identifier: MIT

import argparse
from typing import Dict, Any
from datetime import datetime

from mpp.core.types import VerboseLevel, RawDataFrameColumns as rdc
from mpp.core.views import ViewAggregationLevel
from mpp import __version__
from tools.common.validators.valid_input_types import ValidFileSpec

from .validate import (
    ValidMultiFileSpec,
    ValidInputPathSpec,
    ValidPathSpec,
    ValidSample,
    IntegerRange,
    FloatRange,
    ValidOutputFormat,
    TIMESTAMP_ARG_FORMAT,
    TIMESTAMP_ARG_FORMAT_MS, ValidIntegerRangeList
)

# Constants
MAX_FILE_SIZE_BYTES = 1 * 1024 * 1024  # 1 MB


def parse_args() -> argparse.Namespace:
    """
    Parse command line arguments

    :return: a Namespace object with parsed command line arguments as attributes
    """
    parser = argparse.ArgumentParser(description='Metric post processor', add_help=False,
                                     formatter_class=argparse.RawTextHelpFormatter)

    required_options = parser.add_argument_group('Required options')
    required_options.add_argument('-i', '--input',
                                  dest='input_data_file_path',
                                  metavar='emon.dat or input directory',
                                  type=ValidInputPathSpec(file_must_exist=True),
                                  required=True,
                                  help='input file: input data file',
                                  )
    required_options.add_argument('-m', '--metric',
                                  nargs='*',
                                  dest='metric_file_path',
                                  metavar='metric.xml',
                                  type=ValidPathSpec(file_must_exist=True,
                                                    max_file_size=MAX_FILE_SIZE_BYTES),
                                  required=True,
                                  help='input path: metric definition file or search directory.\n'
                                       'Specify hybrid files using\n'
                                       '<hybrid core type>=<metric definition file>\n'
                                       'If the path is omitted, will attempt to auto-detect file\n'
                                       'in the application directory and parent directory.\n'
                                       'If a directory is used, will attempt to auto-detect file\n'
                                       'in the provided directory.'
                                  )
    required_options.add_argument('-o', '--output',
                                  dest='output_file_specifier',
                                  metavar='excel_file.xlsx | output_directory | prefix',
                                  type=ValidFileSpec(file_must_exist=False,
                                                     file_must_not_exist=False,
                                                     allow_symlinks=False),
                                  required=True,
                                  help='The behavior of this argument is defined by the output\n'
                                       'file specifier as described below. In all cases, the\n'
                                       'output directory is \'.\' if an output path is not\n'
                                       'given.\n\n'
                                       
                                       '  excel_file.xlsx: [directory/]excel_file.xlsx\n'
                                       '    excel_file.xlsx and csv files are written to\n'
                                       '    [directory/]\n\n'

                                       '  output_directory: [output_directory]\n'
                                       '    csv files are written to [directory]\n'
                                       '    If \'--output-format xlsx\' is included on the\n'
                                       '    command line, excel output is written to\n'
                                       '    summary.xlsx in the specified directory\n\n'

                                       '  prefix: [output_directory/]prefix\n'
                                       '    csv files are prefixed with prefix.\n'
                                       '    If \'--output-format xlsx\' is included on the\n'
                                       '    command line, excel output is written to\n'
                                       '    prefix.xlsx in the specified directory\n'
                                  )

    view_options = parser.add_argument_group('View generation options')
    view_options.add_argument('--socket-view',
                              action='store_const',
                              const=ViewAggregationLevel.SOCKET,
                              default=None,
                              help='Generate Socket-level summary and details')
    view_options.add_argument('--die-view',
                              action='store_const',
                              const=ViewAggregationLevel.DIE,
                              default=None,
                              help='Generate Die-level summary and details')
    view_options.add_argument('--core-view',
                              action='store_const',
                              const=ViewAggregationLevel.CORE,
                              default=None,
                              help='Generate Core-level summary and details')
    view_options.add_argument('--thread-view',
                              action='store_const',
                              const=ViewAggregationLevel.THREAD,
                              default=None,
                              help='Generate Thread-level summary and details')
    view_options.add_argument('--uncore-view',
                              action='store_const',
                              const=ViewAggregationLevel.UNCORE,
                              default=None,
                              help='Generate Uncore unit-level summary and details')
    view_options.add_argument('--no-detail-views',
                              action='store_true',
                              default=False,
                              help='Generate only summary views (significantly improves\n'
                                   'performance)')

    optional_options = parser.add_argument_group('Optional options')
    optional_options.add_argument('-j', '--emonv',
                                  dest='emonv_file_path',
                                  metavar='emon-v.dat',
                                  type=ValidFileSpec(file_must_exist=True, max_file_size=MAX_FILE_SIZE_BYTES),
                                  help='input file: emon -v dat',
                                  )
    optional_options.add_argument('-f', '--format',
                                  nargs='*',
                                  dest='chart_format_file_path',
                                  metavar='format.txt',
                                  type=ValidPathSpec(file_must_exist=True, max_file_size=MAX_FILE_SIZE_BYTES),
                                  help='input path: chart format definition file, or search directory\n'
                                       'Specify hybrid files using\n'
                                       '<hybrid core type>=<chart format file>\n'
                                       'If the path is omitted, will attempt to auto-detect file\n'
                                       'in the application directory and parent directory.\n'
                                       'If a directory is used, will attempt to auto-detect file\n'
                                       'in the provided directory.'
                                  )
    optional_options.add_argument('-c', '--frequency',
                                  dest='frequency',
                                  type=IntegerRange(min_value=1),
                                  help='TSC frequency in MHz (e.g., -c 1600)',
                                  )
    optional_options.add_argument('-b', '--begin',
                                  dest='begin_sample',
                                  metavar='#SAMPLE or TIMESTAMP',
                                  type=ValidSample(),
                                  help='First sample to process. Specify a sample number or a\n'
                                       'timestamp (MM/DD/YYYY HH:MM:SS.sss, where sss is\n'
                                       'milliseconds and is optional)',
                                  )
    optional_options.add_argument('-e', '--end',
                                  dest='end_sample',
                                  metavar='#SAMPLE or TIMESTAMP',
                                  type=ValidSample(),
                                  help='Last sample to process. Specify a sample number or a\n'
                                       'timestamp (MM/DD/YYYY HH:MM:SS.sss, where sss is\n'
                                       'milliseconds and is optional)',
                                  )
    optional_options.add_argument('-x', '--tps',
                                  dest='transactions_per_second',
                                  type=FloatRange(),
                                  help='Number of transactions per second for throughput-mode\n'
                                       'reports',
                                  )
    optional_options.add_argument('-l', '--retire-latency',
                                  dest='retire_latency',
                                  metavar='latency.json',
                                  type=ValidMultiFileSpec(file_must_exist=True, max_file_size=MAX_FILE_SIZE_BYTES),
                                  help='retire latency file (json): retire latency definition\n'
                                       'file. Specify hybrid files using\n'
                                       '<hybrid core type>=<retire latency file>',
                                  )
    optional_options.add_argument('--chunk-size',
                                  dest='chunk_size',
                                  type=IntegerRange(),
                                  default=20,
                                  help='Number of event "blocks" to process at a time. Higher\n'
                                       'number requires more memory and may speed up\n'
                                       'processing. Set to 0 to process the entire input file\n'
                                       'in memory (may cause an out of memory error when\n'
                                       'processing large files)',
                                  )
    optional_options.add_argument('-p', '--parallel',
                                  dest='parallel_cores',
                                  metavar='PARALLEL_CORES',
                                  type=IntegerRange(min_value=1),
                                  help='Specify number of cores to run in parallel, if the\n'
                                       'input file size is larger than 6mb. Number of cores\n'
                                       'will default to the minimum of three options:  the\n'
                                       'total number of cores on the system, the number of\n'
                                       'partitions in the file, or 60. If number of cores is 1,\n'
                                       'the post processor will run in serial regardless of\n'
                                       'file size.',
                                  )
    optional_options.add_argument('--percentile',
                                  nargs='?',
                                  const=95,
                                  dest='percentile',
                                  metavar='PERCENTILE',
                                  type=IntegerRange(min_value=1, max_value=99),
                                  help='The percentile to calculate for each metric in the\n'
                                       'system summary. Note: This feature requires optional\n'
                                       'dependencies. '
                                  )
    optional_options.add_argument('--input-pattern',
                                  dest='input_pattern',
                                  metavar='INPUT_PATTERN',
                                  help='Pattern to match input files in a directory. The '
                                       'default pattern is "*.dat"',
                                  type=str,
                                  default='*.dat',)
    optional_options.add_argument('--output-format',
                                  nargs='+',
                                  dest='output_format',
                                  metavar='OUTPUT_FORMAT',
                                  type=ValidOutputFormat(),
                                  help='Additional output formats to write in addition to csv\n'
                                       'output. Supported formats: xlsx',
                                  )
    optional_options.add_argument('--core-filter',
                                    dest='core_filter',
                                    metavar='CORE_FILTER',
                                    type=ValidIntegerRangeList(),
                                    help='Determines the specific cores to process, as specified by the core\'s OS '
                                         'Processor value. By default, all cores are processed.\n'
                                         'Example: --core-filter 0-3,6,8-10 will process cores with OS Processor values '
                                         '0, 1, 2, 3, 6, 8, 9, and 10.\n',
                                    default=None,
                                    )
    comparison_options = parser.add_argument_group('Mpp Comparison options')
    comparison_options.add_argument('--compare',
                                  nargs='?',
                                  const='comparison.xlsx',
                                  default=None,
                                  metavar='COMPARISON.xlsx',
                                  type=str,
                                  help='Compare multiple files after running batch processing. Currently supports '
                                   'comparisons in the system summary views only'
                                  )
    comparison_options.add_argument('--baseline',
                              help='The baseline file to compare against',
                              type=str,
                              metavar='BASELINE_FILE',
                              )
    comparison_options.add_argument('--delta',
                              help='The percentage difference that will flag compared summary values',
                              type=FloatRange(max_value=100),
                              metavar='DELTA %',
                              default=5)

    misc_options = parser.add_argument_group('Miscellaneous options')
    misc_options.add_argument('-h', '--help',
                              help='Show this help message and exit',
                              action='help')
    misc_options.add_argument('--version',
                              help='Show version information and exit',
                              action='version',
                              version=f'%(prog)s {__version__}')
    misc_options.add_argument('--verbose',
                              help='Increase output verbosity. Options are FULL and DEBUG. FULL means full regular '
                                   'output for batch processing, and DEBUG includes debug output for potential metric errors',
                              type=str,
                              choices=[VerboseLevel.BATCH_FULL, VerboseLevel.DEBUG],
                              default=VerboseLevel.INFO)
    misc_options.add_argument('--disable-polars',
                              help='Disables polars CSV writing and reverts to writing CSV files with pandas',
                              action='store_true')
    misc_options.add_argument('--disable-rust',
                              help='Disables rust based pathways for faster performance',
                              action='store_true')
    misc_options.add_argument('--recursive',
                              help='Recursively search for input files in directories',
                              action='store_true')

    hidden_options = parser.add_argument_group('Hidden options')
    hidden_options.add_argument('--force-parallel',
                                help=argparse.SUPPRESS,
                                action='store_true')
    hidden_options.add_argument('-p-per-job', '--parallel-per-job',
                                  dest='parallel_cores_per_runner',
                                  help=argparse.SUPPRESS,
                                  metavar='PARALLEL_CORES_PER_JOB',
                                  type=IntegerRange(min_value=0),
                                  default=0
                                )
    return _validate_and_parse_args(parser)


def get_sample_range_args(args: argparse.Namespace) -> Dict[str, Any]:
    if type(args.begin_sample) is datetime or type(args.end_sample) is datetime:
        return {'from_timestamp': args.begin_sample, 'to_timestamp': args.end_sample}
    else:
        return {'from_sample': args.begin_sample, 'to_sample': args.end_sample}





def reformat_args_for_consumption(args):
    """
    Reformat argparse Namespace arguments for easier downstream consumption.
    Uses direct attribute access for maintainability and extensibility.
    """
    from .format import MultiFileSpecFormatter, IntegerRangeListFormatter

    # Multi-file arguments
    if args.metric_file_path is not None:
        args.metric_file_path = MultiFileSpecFormatter.reformat_multi_file_args(args.metric_file_path)
    if args.chart_format_file_path is not None:
        args.chart_format_file_path = MultiFileSpecFormatter.reformat_multi_file_args(args.chart_format_file_path)

    # Integer range list arguments
    if hasattr(args, 'core_filter'):
        integer_range_list_formatter = IntegerRangeListFormatter(rdc.CORE)
        args.core_filter = integer_range_list_formatter.format(args.core_filter)
    # Add more argument/formatter pairs here as needed, using direct attribute access

    return args


def _validate_and_parse_args(parser: argparse.ArgumentParser) -> argparse.Namespace:
    """
    Parse and validate command line arguments. Terminate the program if there are errors

    :param parser: the argument parser

    :return: a Namespace object with parsed arguments as attributes
    """

    args = parser.parse_args()
    args = reformat_args_for_consumption(args)
    if args.begin_sample and args.end_sample and type(args.begin_sample) is not type(args.end_sample):
        parser.error('please specify either timestamps or sample numbers for "begin sample" and "end sample"')
    if args.begin_sample and args.end_sample and args.begin_sample > args.end_sample:
        parser.error('the "begin sample" value cannot be greater than the "end sample" value')
    return args
