﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/DateTime.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/redshift/RedshiftRequest.h>
#include <aws/redshift/Redshift_EXPORTS.h>
#include <aws/redshift/model/ScheduledActionType.h>

#include <utility>

namespace Aws {
namespace Redshift {
namespace Model {

/**
 */
class CreateScheduledActionRequest : public RedshiftRequest {
 public:
  AWS_REDSHIFT_API CreateScheduledActionRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "CreateScheduledAction"; }

  AWS_REDSHIFT_API Aws::String SerializePayload() const override;

 protected:
  AWS_REDSHIFT_API void DumpBodyToUrl(Aws::Http::URI& uri) const override;

 public:
  ///@{
  /**
   * <p>The name of the scheduled action. The name must be unique within an account.
   * For more information about this parameter, see <a>ScheduledAction</a>. </p>
   */
  inline const Aws::String& GetScheduledActionName() const { return m_scheduledActionName; }
  inline bool ScheduledActionNameHasBeenSet() const { return m_scheduledActionNameHasBeenSet; }
  template <typename ScheduledActionNameT = Aws::String>
  void SetScheduledActionName(ScheduledActionNameT&& value) {
    m_scheduledActionNameHasBeenSet = true;
    m_scheduledActionName = std::forward<ScheduledActionNameT>(value);
  }
  template <typename ScheduledActionNameT = Aws::String>
  CreateScheduledActionRequest& WithScheduledActionName(ScheduledActionNameT&& value) {
    SetScheduledActionName(std::forward<ScheduledActionNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A JSON format string of the Amazon Redshift API operation with input
   * parameters. For more information about this parameter, see
   * <a>ScheduledAction</a>. </p>
   */
  inline const ScheduledActionType& GetTargetAction() const { return m_targetAction; }
  inline bool TargetActionHasBeenSet() const { return m_targetActionHasBeenSet; }
  template <typename TargetActionT = ScheduledActionType>
  void SetTargetAction(TargetActionT&& value) {
    m_targetActionHasBeenSet = true;
    m_targetAction = std::forward<TargetActionT>(value);
  }
  template <typename TargetActionT = ScheduledActionType>
  CreateScheduledActionRequest& WithTargetAction(TargetActionT&& value) {
    SetTargetAction(std::forward<TargetActionT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The schedule in <code>at( )</code> or <code>cron( )</code> format. For more
   * information about this parameter, see <a>ScheduledAction</a>.</p>
   */
  inline const Aws::String& GetSchedule() const { return m_schedule; }
  inline bool ScheduleHasBeenSet() const { return m_scheduleHasBeenSet; }
  template <typename ScheduleT = Aws::String>
  void SetSchedule(ScheduleT&& value) {
    m_scheduleHasBeenSet = true;
    m_schedule = std::forward<ScheduleT>(value);
  }
  template <typename ScheduleT = Aws::String>
  CreateScheduledActionRequest& WithSchedule(ScheduleT&& value) {
    SetSchedule(std::forward<ScheduleT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The IAM role to assume to run the target action. For more information about
   * this parameter, see <a>ScheduledAction</a>. </p>
   */
  inline const Aws::String& GetIamRole() const { return m_iamRole; }
  inline bool IamRoleHasBeenSet() const { return m_iamRoleHasBeenSet; }
  template <typename IamRoleT = Aws::String>
  void SetIamRole(IamRoleT&& value) {
    m_iamRoleHasBeenSet = true;
    m_iamRole = std::forward<IamRoleT>(value);
  }
  template <typename IamRoleT = Aws::String>
  CreateScheduledActionRequest& WithIamRole(IamRoleT&& value) {
    SetIamRole(std::forward<IamRoleT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The description of the scheduled action. </p>
   */
  inline const Aws::String& GetScheduledActionDescription() const { return m_scheduledActionDescription; }
  inline bool ScheduledActionDescriptionHasBeenSet() const { return m_scheduledActionDescriptionHasBeenSet; }
  template <typename ScheduledActionDescriptionT = Aws::String>
  void SetScheduledActionDescription(ScheduledActionDescriptionT&& value) {
    m_scheduledActionDescriptionHasBeenSet = true;
    m_scheduledActionDescription = std::forward<ScheduledActionDescriptionT>(value);
  }
  template <typename ScheduledActionDescriptionT = Aws::String>
  CreateScheduledActionRequest& WithScheduledActionDescription(ScheduledActionDescriptionT&& value) {
    SetScheduledActionDescription(std::forward<ScheduledActionDescriptionT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The start time in UTC of the scheduled action. Before this time, the
   * scheduled action does not trigger. For more information about this parameter,
   * see <a>ScheduledAction</a>.</p>
   */
  inline const Aws::Utils::DateTime& GetStartTime() const { return m_startTime; }
  inline bool StartTimeHasBeenSet() const { return m_startTimeHasBeenSet; }
  template <typename StartTimeT = Aws::Utils::DateTime>
  void SetStartTime(StartTimeT&& value) {
    m_startTimeHasBeenSet = true;
    m_startTime = std::forward<StartTimeT>(value);
  }
  template <typename StartTimeT = Aws::Utils::DateTime>
  CreateScheduledActionRequest& WithStartTime(StartTimeT&& value) {
    SetStartTime(std::forward<StartTimeT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The end time in UTC of the scheduled action. After this time, the scheduled
   * action does not trigger. For more information about this parameter, see
   * <a>ScheduledAction</a>. </p>
   */
  inline const Aws::Utils::DateTime& GetEndTime() const { return m_endTime; }
  inline bool EndTimeHasBeenSet() const { return m_endTimeHasBeenSet; }
  template <typename EndTimeT = Aws::Utils::DateTime>
  void SetEndTime(EndTimeT&& value) {
    m_endTimeHasBeenSet = true;
    m_endTime = std::forward<EndTimeT>(value);
  }
  template <typename EndTimeT = Aws::Utils::DateTime>
  CreateScheduledActionRequest& WithEndTime(EndTimeT&& value) {
    SetEndTime(std::forward<EndTimeT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>If true, the schedule is enabled. If false, the scheduled action does not
   * trigger. For more information about <code>state</code> of the scheduled action,
   * see <a>ScheduledAction</a>. </p>
   */
  inline bool GetEnable() const { return m_enable; }
  inline bool EnableHasBeenSet() const { return m_enableHasBeenSet; }
  inline void SetEnable(bool value) {
    m_enableHasBeenSet = true;
    m_enable = value;
  }
  inline CreateScheduledActionRequest& WithEnable(bool value) {
    SetEnable(value);
    return *this;
  }
  ///@}
 private:
  Aws::String m_scheduledActionName;

  ScheduledActionType m_targetAction;

  Aws::String m_schedule;

  Aws::String m_iamRole;

  Aws::String m_scheduledActionDescription;

  Aws::Utils::DateTime m_startTime{};

  Aws::Utils::DateTime m_endTime{};

  bool m_enable{false};
  bool m_scheduledActionNameHasBeenSet = false;
  bool m_targetActionHasBeenSet = false;
  bool m_scheduleHasBeenSet = false;
  bool m_iamRoleHasBeenSet = false;
  bool m_scheduledActionDescriptionHasBeenSet = false;
  bool m_startTimeHasBeenSet = false;
  bool m_endTimeHasBeenSet = false;
  bool m_enableHasBeenSet = false;
};

}  // namespace Model
}  // namespace Redshift
}  // namespace Aws
