/*
 * Decompiled with CFR 0.152.
 */
package org.duckdb;

import java.math.BigDecimal;
import java.math.BigInteger;
import java.nio.ByteBuffer;
import java.nio.DoubleBuffer;
import java.nio.FloatBuffer;
import java.nio.IntBuffer;
import java.nio.LongBuffer;
import java.nio.ShortBuffer;
import java.nio.charset.StandardCharsets;
import java.sql.SQLException;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.LocalTime;
import java.time.OffsetDateTime;
import java.time.OffsetTime;
import java.time.ZoneOffset;
import java.time.ZonedDateTime;
import java.time.temporal.ChronoUnit;
import java.util.AbstractMap;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Date;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.UUID;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import org.duckdb.DuckDBBindings;
import org.duckdb.DuckDBConnection;
import org.duckdb.DuckDBHugeInt;

public class DuckDBAppender
implements AutoCloseable {
    private static final Set<Integer> supportedTypes = new LinkedHashSet<Integer>();
    private static final DuckDBBindings.CAPIType[] int8Types;
    private static final DuckDBBindings.CAPIType[] int16Types;
    private static final DuckDBBindings.CAPIType[] int32Types;
    private static final DuckDBBindings.CAPIType[] int64Types;
    private static final DuckDBBindings.CAPIType[] int128Types;
    private static final DuckDBBindings.CAPIType[] timestampLocalTypes;
    private static final DuckDBBindings.CAPIType[] timestampMicrosTypes;
    private static final DuckDBBindings.CAPIType[] collectionTypes;
    private static final int STRING_MAX_INLINE_BYTES = 12;
    private static final LocalDateTime EPOCH_DATE_TIME;
    private final DuckDBConnection conn;
    private final String catalog;
    private final String schema;
    private final String table;
    private final long maxRows;
    private ByteBuffer appenderRef;
    private final Lock appenderRefLock = new ReentrantLock();
    private final ByteBuffer chunkRef;
    private final List<Column> columns;
    private long rowIdx = 0L;
    private Column currentColumn = null;
    private Column prevColumn = null;
    private boolean writeInlinedStrings = true;

    DuckDBAppender(DuckDBConnection conn, String catalog, String schema, String table) throws SQLException {
        this.conn = conn;
        this.catalog = catalog;
        this.schema = schema;
        this.table = table;
        this.maxRows = DuckDBBindings.duckdb_vector_size();
        ByteBuffer appenderRef = null;
        ByteBuffer[] colTypes = null;
        ByteBuffer chunkRef = null;
        List<Column> cols = null;
        try {
            appenderRef = DuckDBAppender.createAppender(conn, catalog, schema, table);
            colTypes = DuckDBAppender.readTableTypes(appenderRef);
            chunkRef = DuckDBAppender.createChunk(colTypes);
            cols = DuckDBAppender.createTopLevelColumns(chunkRef, colTypes);
        }
        catch (Exception e) {
            if (null != chunkRef) {
                DuckDBBindings.duckdb_destroy_data_chunk(chunkRef);
            }
            if (null != colTypes) {
                for (ByteBuffer ct : colTypes) {
                    if (null == ct) continue;
                    DuckDBBindings.duckdb_destroy_logical_type(ct);
                }
            }
            if (null != appenderRef) {
                DuckDBBindings.duckdb_appender_destroy(appenderRef);
            }
            throw new SQLException(this.createErrMsg(e.getMessage()), e);
        }
        this.appenderRef = appenderRef;
        this.chunkRef = chunkRef;
        this.columns = cols;
    }

    public DuckDBAppender beginRow() throws SQLException {
        this.checkOpen();
        if (!this.readyForANewRowInvariant()) {
            throw new SQLException(this.createErrMsg("'endRow' must be called before calling 'beginRow' again"));
        }
        if (null == this.columns || 0 == this.columns.size()) {
            throw new SQLException(this.createErrMsg("no columns found to append to"));
        }
        this.currentColumn = this.columns.get(0);
        return this;
    }

    public DuckDBAppender endRow() throws SQLException {
        this.checkOpen();
        if (!this.rowCompletedInvariant()) {
            Column topCol = this.currentTopLevelColumn();
            if (null != topCol) {
                throw new SQLException(this.createErrMsg("all columns must be appended to before calling 'endRow', expected columns count: " + this.columns.size() + ", actual: " + (topCol.idx + 1)));
            }
            throw new SQLException(this.createErrMsg("calls to 'beginRow' and 'endRow' must be paired and cannot be interleaved with other 'begin*' and 'end*' calls"));
        }
        ++this.rowIdx;
        Column prev = this.prevColumn;
        this.prevColumn = null;
        if (this.rowIdx >= this.maxRows) {
            try {
                this.flush();
            }
            catch (SQLException e) {
                this.prevColumn = prev;
                --this.rowIdx;
                throw e;
            }
        }
        return this;
    }

    public DuckDBAppender beginStruct() throws SQLException {
        this.checkOpen();
        if (!this.rowBegunInvariant()) {
            throw new SQLException(this.createErrMsg("'beginRow' must be called before calling 'beginStruct'"));
        }
        this.checkCurrentColumnType(DuckDBBindings.CAPIType.DUCKDB_TYPE_STRUCT);
        if (0 == this.currentColumn.children.size()) {
            throw new SQLException(this.createErrMsg("invalid empty struct"));
        }
        this.currentColumn = (Column)this.currentColumn.children.get(0);
        return this;
    }

    public DuckDBAppender endStruct() throws SQLException {
        this.checkOpen();
        if (!this.structCompletedInvariant()) {
            if (this.structBegunInvariant()) {
                throw new SQLException(this.createErrMsg("all struct fields must be appended to before calling 'endStruct', expected fields count: " + this.currentColumn.parent.children.size() + ", actual: " + (this.currentColumn.idx + 1)));
            }
            throw new SQLException(this.createErrMsg("all struct fields must be appended to before calling 'endStruct'"));
        }
        this.prevColumn = this.prevColumn.parent;
        return this;
    }

    public DuckDBAppender beginUnion(String tag) throws SQLException {
        this.checkOpen();
        if (!this.rowBegunInvariant()) {
            throw new SQLException(this.createErrMsg("'beginRow' must be called before calling 'beginUnion'"));
        }
        Column col = this.currentColumn(DuckDBBindings.CAPIType.DUCKDB_TYPE_UNION);
        this.currentColumn = this.putUnionTag(col, this.rowIdx, tag);
        return this;
    }

    public DuckDBAppender endUnion() throws SQLException {
        this.checkOpen();
        if (!this.unionCompletedInvariant()) {
            throw new SQLException(this.createErrMsg("union column must be appended to before calling 'endUnion'"));
        }
        this.prevColumn = this.prevColumn.parent;
        return this;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public long flush() throws SQLException {
        this.checkOpen();
        if (!this.readyForANewRowInvariant()) {
            throw new SQLException(this.createErrMsg("'endRow' must be called before calling 'flush'"));
        }
        if (0L == this.rowIdx) {
            return this.rowIdx;
        }
        this.appenderRefLock.lock();
        try {
            this.checkOpen();
            DuckDBBindings.duckdb_data_chunk_set_size(this.chunkRef, this.rowIdx);
            int appendState = DuckDBBindings.duckdb_append_data_chunk(this.appenderRef, this.chunkRef);
            if (0 != appendState) {
                byte[] errorUTF8 = DuckDBBindings.duckdb_appender_error(this.appenderRef);
                String error = DuckDBAppender.strFromUTF8(errorUTF8);
                throw new SQLException(this.createErrMsg(error));
            }
            int flushState = DuckDBBindings.duckdb_appender_flush(this.appenderRef);
            if (0 != flushState) {
                byte[] errorUTF8 = DuckDBBindings.duckdb_appender_error(this.appenderRef);
                String error = DuckDBAppender.strFromUTF8(errorUTF8);
                throw new SQLException(this.createErrMsg(error));
            }
            DuckDBBindings.duckdb_data_chunk_reset(this.chunkRef);
            try {
                for (Column col : this.columns) {
                    col.reset();
                }
            }
            catch (SQLException e) {
                throw new SQLException(this.createErrMsg(e.getMessage()), e);
            }
            long ret = this.rowIdx;
            this.rowIdx = 0L;
            long l = ret;
            return l;
        }
        finally {
            this.appenderRefLock.unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void close() throws SQLException {
        if (this.isClosed()) {
            return;
        }
        this.appenderRefLock.lock();
        try {
            if (this.isClosed()) {
                return;
            }
            if (this.rowIdx > 0L) {
                try {
                    this.flush();
                }
                catch (SQLException sQLException) {
                    // empty catch block
                }
            }
            for (Column col : this.columns) {
                col.destroy();
            }
            DuckDBBindings.duckdb_destroy_data_chunk(this.chunkRef);
            DuckDBBindings.duckdb_appender_close(this.appenderRef);
            DuckDBBindings.duckdb_appender_destroy(this.appenderRef);
            if (!this.conn.closing) {
                this.conn.connRefLock.lock();
                try {
                    this.conn.appenders.remove(this);
                }
                finally {
                    this.conn.connRefLock.unlock();
                }
            }
            this.appenderRef = null;
        }
        finally {
            this.appenderRefLock.unlock();
        }
    }

    public boolean isClosed() throws SQLException {
        return this.appenderRef == null;
    }

    public DuckDBAppender append(boolean value) throws SQLException {
        Column col = this.currentColumn(DuckDBBindings.CAPIType.DUCKDB_TYPE_BOOLEAN);
        byte val = (byte)(value ? 1 : 0);
        this.putByte(col, this.rowIdx, val);
        this.moveToNextColumn();
        return this;
    }

    public DuckDBAppender append(char value) throws SQLException {
        String str = String.valueOf(value);
        return this.append(str);
    }

    public DuckDBAppender append(byte value) throws SQLException {
        Column col = this.currentColumn(int8Types);
        this.putByte(col, this.rowIdx, value);
        this.moveToNextColumn();
        return this;
    }

    public DuckDBAppender append(short value) throws SQLException {
        Column col = this.currentColumn(int16Types);
        this.putShort(col, this.rowIdx, value);
        this.moveToNextColumn();
        return this;
    }

    public DuckDBAppender append(int value) throws SQLException {
        Column col = this.currentColumn(int32Types);
        this.putInt(col, this.rowIdx, value);
        this.moveToNextColumn();
        return this;
    }

    public DuckDBAppender append(long value) throws SQLException {
        Column col = this.currentColumn(int64Types);
        this.putLong(col, this.rowIdx, value);
        this.moveToNextColumn();
        return this;
    }

    public DuckDBAppender append(float value) throws SQLException {
        Column col = this.currentColumn(DuckDBBindings.CAPIType.DUCKDB_TYPE_FLOAT);
        this.putFloat(col, this.rowIdx, value);
        this.moveToNextColumn();
        return this;
    }

    public DuckDBAppender append(double value) throws SQLException {
        Column col = this.currentColumn(DuckDBBindings.CAPIType.DUCKDB_TYPE_DOUBLE);
        this.putDouble(col, this.rowIdx, value);
        this.moveToNextColumn();
        return this;
    }

    public DuckDBAppender append(Boolean value) throws SQLException {
        this.currentColumn(DuckDBBindings.CAPIType.DUCKDB_TYPE_BOOLEAN);
        if (value == null) {
            return this.appendNull();
        }
        return this.append((boolean)value);
    }

    public DuckDBAppender append(Character value) throws SQLException {
        this.currentColumn(DuckDBBindings.CAPIType.DUCKDB_TYPE_VARCHAR);
        if (value == null) {
            return this.appendNull();
        }
        return this.append(value.charValue());
    }

    public DuckDBAppender append(Byte value) throws SQLException {
        this.currentColumn(int8Types);
        if (value == null) {
            return this.appendNull();
        }
        return this.append((byte)value);
    }

    public DuckDBAppender append(Short value) throws SQLException {
        this.currentColumn(int16Types);
        if (value == null) {
            return this.appendNull();
        }
        return this.append((short)value);
    }

    public DuckDBAppender append(Integer value) throws SQLException {
        this.currentColumn(int32Types);
        if (value == null) {
            return this.appendNull();
        }
        return this.append((int)value);
    }

    public DuckDBAppender append(Long value) throws SQLException {
        this.currentColumn(int64Types);
        if (value == null) {
            return this.appendNull();
        }
        return this.append((long)value);
    }

    public DuckDBAppender append(Float value) throws SQLException {
        this.currentColumn(DuckDBBindings.CAPIType.DUCKDB_TYPE_FLOAT);
        if (value == null) {
            return this.appendNull();
        }
        return this.append(value.floatValue());
    }

    public DuckDBAppender append(Double value) throws SQLException {
        this.currentColumn(DuckDBBindings.CAPIType.DUCKDB_TYPE_DOUBLE);
        if (value == null) {
            return this.appendNull();
        }
        return this.append((double)value);
    }

    public DuckDBAppender appendHugeInt(long lower, long upper) throws SQLException {
        Column col = this.currentColumn(int128Types);
        this.putHugeInt(col, this.rowIdx, lower, upper);
        this.moveToNextColumn();
        return this;
    }

    public DuckDBAppender append(BigInteger value) throws SQLException {
        Column col = this.currentColumn(int128Types);
        if (value == null) {
            return this.appendNull();
        }
        this.putBigInteger(col, this.rowIdx, value);
        this.moveToNextColumn();
        return this;
    }

    public DuckDBAppender appendDecimal(short value) throws SQLException {
        Column col = this.currentColumn(DuckDBBindings.CAPIType.DUCKDB_TYPE_DECIMAL);
        this.checkDecimalType(col, DuckDBBindings.CAPIType.DUCKDB_TYPE_SMALLINT);
        this.putDecimal(col, this.rowIdx, value);
        this.moveToNextColumn();
        return this;
    }

    public DuckDBAppender appendDecimal(int value) throws SQLException {
        Column col = this.currentColumn(DuckDBBindings.CAPIType.DUCKDB_TYPE_DECIMAL);
        this.checkDecimalType(col, DuckDBBindings.CAPIType.DUCKDB_TYPE_INTEGER);
        this.putDecimal(col, this.rowIdx, value);
        this.moveToNextColumn();
        return this;
    }

    public DuckDBAppender appendDecimal(long value) throws SQLException {
        Column col = this.currentColumn(DuckDBBindings.CAPIType.DUCKDB_TYPE_DECIMAL);
        this.checkDecimalType(col, DuckDBBindings.CAPIType.DUCKDB_TYPE_BIGINT);
        this.putDecimal(col, this.rowIdx, value);
        this.moveToNextColumn();
        return this;
    }

    public DuckDBAppender appendDecimal(long lower, long upper) throws SQLException {
        Column col = this.currentColumn(DuckDBBindings.CAPIType.DUCKDB_TYPE_DECIMAL);
        this.checkDecimalType(col, DuckDBBindings.CAPIType.DUCKDB_TYPE_HUGEINT);
        this.putDecimal(col, this.rowIdx, lower, upper);
        this.moveToNextColumn();
        return this;
    }

    public DuckDBAppender append(BigDecimal value) throws SQLException {
        Column col = this.currentColumn(DuckDBBindings.CAPIType.DUCKDB_TYPE_DECIMAL);
        if (value == null) {
            return this.appendNull();
        }
        this.putDecimal(col, this.rowIdx, value);
        this.moveToNextColumn();
        return this;
    }

    public DuckDBAppender append(boolean[] values) throws SQLException {
        return this.append(values, null);
    }

    public DuckDBAppender append(boolean[] values, boolean[] nullMask) throws SQLException {
        Column col = this.currentColumn(collectionTypes);
        this.arrayInnerColumn(col, DuckDBBindings.CAPIType.DUCKDB_TYPE_BOOLEAN);
        if (values == null) {
            return this.appendNull();
        }
        this.putBoolArray(col, this.rowIdx, values, nullMask);
        this.moveToNextColumn();
        return this;
    }

    public DuckDBAppender append(boolean[][] values) throws SQLException {
        return this.append(values, (boolean[][])null);
    }

    public DuckDBAppender append(boolean[][] values, boolean[][] nullMask) throws SQLException {
        Column col = this.currentColumn(DuckDBBindings.CAPIType.DUCKDB_TYPE_ARRAY);
        Column inner = this.arrayInnerColumn(col, DuckDBBindings.CAPIType.DUCKDB_TYPE_ARRAY);
        this.arrayInnerColumn(inner, DuckDBBindings.CAPIType.DUCKDB_TYPE_BOOLEAN);
        if (values == null) {
            return this.appendNull();
        }
        this.putBoolArray2D(col, this.rowIdx, values, nullMask);
        this.moveToNextColumn();
        return this;
    }

    public DuckDBAppender appendByteArray(byte[] values) throws SQLException {
        return this.appendByteArray(values, null);
    }

    public DuckDBAppender appendByteArray(byte[] values, boolean[] nullMask) throws SQLException {
        Column col = this.currentColumn(collectionTypes);
        this.arrayInnerColumn(col, int8Types);
        if (values == null) {
            return this.appendNull();
        }
        this.putByteArray(col, this.rowIdx, values, nullMask);
        this.moveToNextColumn();
        return this;
    }

    public DuckDBAppender appendByteArray(byte[][] values) throws SQLException {
        return this.appendByteArray(values, (boolean[][])null);
    }

    public DuckDBAppender appendByteArray(byte[][] values, boolean[][] nullMask) throws SQLException {
        Column col = this.currentColumn(DuckDBBindings.CAPIType.DUCKDB_TYPE_ARRAY);
        Column inner = this.arrayInnerColumn(col, DuckDBBindings.CAPIType.DUCKDB_TYPE_ARRAY);
        this.arrayInnerColumn(inner, int8Types);
        if (values == null) {
            return this.appendNull();
        }
        this.putByteArray2D(col, this.rowIdx, values, nullMask);
        this.moveToNextColumn();
        return this;
    }

    public DuckDBAppender append(byte[] values) throws SQLException {
        Column col = this.currentColumn(DuckDBBindings.CAPIType.DUCKDB_TYPE_BLOB);
        if (values == null) {
            return this.appendNull();
        }
        this.putStringOrBlob(col, this.rowIdx, values);
        this.moveToNextColumn();
        return this;
    }

    public DuckDBAppender append(char[] characters) throws SQLException {
        this.checkCurrentColumnType(DuckDBBindings.CAPIType.DUCKDB_TYPE_VARCHAR);
        if (characters == null) {
            return this.appendNull();
        }
        String str = String.valueOf(characters);
        return this.append(str);
    }

    public DuckDBAppender append(short[] values) throws SQLException {
        return this.append(values, (boolean[])null);
    }

    public DuckDBAppender append(short[] values, boolean[] nullMask) throws SQLException {
        Column col = this.currentColumn(collectionTypes);
        this.arrayInnerColumn(col, int16Types);
        if (values == null) {
            return this.appendNull();
        }
        this.putShortArray(col, this.rowIdx, values, nullMask);
        this.moveToNextColumn();
        return this;
    }

    public DuckDBAppender append(short[][] values) throws SQLException {
        return this.append(values, (boolean[][])null);
    }

    public DuckDBAppender append(short[][] values, boolean[][] nullMask) throws SQLException {
        Column col = this.currentColumn(DuckDBBindings.CAPIType.DUCKDB_TYPE_ARRAY);
        Column inner = this.arrayInnerColumn(col, DuckDBBindings.CAPIType.DUCKDB_TYPE_ARRAY);
        this.arrayInnerColumn(inner, int16Types);
        if (values == null) {
            return this.appendNull();
        }
        this.putShortArray2D(col, this.rowIdx, values, nullMask);
        this.moveToNextColumn();
        return this;
    }

    public DuckDBAppender append(int[] values) throws SQLException {
        return this.append(values, (boolean[])null);
    }

    public DuckDBAppender append(int[] values, boolean[] nullMask) throws SQLException {
        Column col = this.currentColumn(collectionTypes);
        this.arrayInnerColumn(col, int32Types);
        if (values == null) {
            return this.appendNull();
        }
        this.putIntArray(col, this.rowIdx, values, nullMask);
        this.moveToNextColumn();
        return this;
    }

    public DuckDBAppender append(int[][] values) throws SQLException {
        return this.append(values, (boolean[][])null);
    }

    public DuckDBAppender append(int[][] values, boolean[][] nullMask) throws SQLException {
        Column col = this.currentColumn(DuckDBBindings.CAPIType.DUCKDB_TYPE_ARRAY);
        Column inner = this.arrayInnerColumn(col, DuckDBBindings.CAPIType.DUCKDB_TYPE_ARRAY);
        this.arrayInnerColumn(inner, int32Types);
        if (values == null) {
            return this.appendNull();
        }
        this.putIntArray2D(col, this.rowIdx, values, nullMask);
        this.moveToNextColumn();
        return this;
    }

    public DuckDBAppender append(long[] values) throws SQLException {
        return this.append(values, (boolean[])null);
    }

    public DuckDBAppender append(long[] values, boolean[] nullMask) throws SQLException {
        Column col = this.currentColumn(collectionTypes);
        this.arrayInnerColumn(col, int64Types);
        if (values == null) {
            return this.appendNull();
        }
        this.putLongArray(col, this.rowIdx, values, nullMask);
        this.moveToNextColumn();
        return this;
    }

    public DuckDBAppender append(long[][] values) throws SQLException {
        return this.append(values, (boolean[][])null);
    }

    public DuckDBAppender append(long[][] values, boolean[][] nullMask) throws SQLException {
        Column col = this.currentColumn(DuckDBBindings.CAPIType.DUCKDB_TYPE_ARRAY);
        Column inner = this.arrayInnerColumn(col, DuckDBBindings.CAPIType.DUCKDB_TYPE_ARRAY);
        this.arrayInnerColumn(inner, int64Types);
        if (values == null) {
            return this.appendNull();
        }
        this.putLongArray2D(col, this.rowIdx, values, nullMask);
        this.moveToNextColumn();
        return this;
    }

    public DuckDBAppender append(float[] values) throws SQLException {
        return this.append(values, (boolean[])null);
    }

    public DuckDBAppender append(float[] values, boolean[] nullMask) throws SQLException {
        Column col = this.currentColumn(collectionTypes);
        this.arrayInnerColumn(col, DuckDBBindings.CAPIType.DUCKDB_TYPE_FLOAT);
        if (values == null) {
            return this.appendNull();
        }
        this.putFloatArray(col, this.rowIdx, values, nullMask);
        this.moveToNextColumn();
        return this;
    }

    public DuckDBAppender append(float[][] values) throws SQLException {
        return this.append(values, (boolean[][])null);
    }

    public DuckDBAppender append(float[][] values, boolean[][] nullMask) throws SQLException {
        Column col = this.currentColumn(DuckDBBindings.CAPIType.DUCKDB_TYPE_ARRAY);
        Column inner = this.arrayInnerColumn(col, DuckDBBindings.CAPIType.DUCKDB_TYPE_ARRAY);
        this.arrayInnerColumn(inner, DuckDBBindings.CAPIType.DUCKDB_TYPE_FLOAT);
        if (values == null) {
            return this.appendNull();
        }
        this.putFloatArray2D(col, this.rowIdx, values, nullMask);
        this.moveToNextColumn();
        return this;
    }

    public DuckDBAppender append(double[] values) throws SQLException {
        return this.append(values, null);
    }

    public DuckDBAppender append(double[] values, boolean[] nullMask) throws SQLException {
        Column col = this.currentColumn(collectionTypes);
        this.arrayInnerColumn(col, DuckDBBindings.CAPIType.DUCKDB_TYPE_DOUBLE);
        if (values == null) {
            return this.appendNull();
        }
        this.putDoubleArray(col, this.rowIdx, values, nullMask);
        this.moveToNextColumn();
        return this;
    }

    public DuckDBAppender append(double[][] values) throws SQLException {
        return this.append(values, (boolean[][])null);
    }

    public DuckDBAppender append(double[][] values, boolean[][] nullMask) throws SQLException {
        Column col = this.currentColumn(DuckDBBindings.CAPIType.DUCKDB_TYPE_ARRAY);
        Column inner = this.arrayInnerColumn(col, DuckDBBindings.CAPIType.DUCKDB_TYPE_ARRAY);
        this.arrayInnerColumn(inner, DuckDBBindings.CAPIType.DUCKDB_TYPE_DOUBLE);
        if (values == null) {
            return this.appendNull();
        }
        this.putDoubleArray2D(col, this.rowIdx, values, nullMask);
        this.moveToNextColumn();
        return this;
    }

    public DuckDBAppender append(String value) throws SQLException {
        Column col = this.currentColumn(DuckDBBindings.CAPIType.DUCKDB_TYPE_VARCHAR);
        if (value == null) {
            return this.appendNull();
        }
        byte[] bytes = value.getBytes(StandardCharsets.UTF_8);
        this.putStringOrBlob(col, this.rowIdx, bytes);
        this.moveToNextColumn();
        return this;
    }

    public DuckDBAppender appendUUID(long mostSigBits, long leastSigBits) throws SQLException {
        Column col = this.currentColumn(DuckDBBindings.CAPIType.DUCKDB_TYPE_UUID);
        this.putUUID(col, this.rowIdx, mostSigBits, leastSigBits);
        this.moveToNextColumn();
        return this;
    }

    public DuckDBAppender append(UUID value) throws SQLException {
        this.currentColumn(DuckDBBindings.CAPIType.DUCKDB_TYPE_UUID);
        if (value == null) {
            return this.appendNull();
        }
        long mostSigBits = value.getMostSignificantBits();
        long leastSigBits = value.getLeastSignificantBits();
        return this.appendUUID(mostSigBits, leastSigBits);
    }

    public DuckDBAppender appendEpochDays(int days) throws SQLException {
        Column col = this.currentColumn(DuckDBBindings.CAPIType.DUCKDB_TYPE_DATE);
        this.putEpochDays(col, this.rowIdx, days);
        this.moveToNextColumn();
        return this;
    }

    public DuckDBAppender append(LocalDate value) throws SQLException {
        Column col = this.currentColumn(DuckDBBindings.CAPIType.DUCKDB_TYPE_DATE);
        if (value == null) {
            return this.appendNull();
        }
        this.putLocalDate(col, this.rowIdx, value);
        this.moveToNextColumn();
        return this;
    }

    public DuckDBAppender appendDayMicros(long micros) throws SQLException {
        Column col = this.currentColumn(DuckDBBindings.CAPIType.DUCKDB_TYPE_TIME);
        this.putDayMicros(col, this.rowIdx, micros);
        this.moveToNextColumn();
        return this;
    }

    public DuckDBAppender append(LocalTime value) throws SQLException {
        Column col = this.currentColumn(DuckDBBindings.CAPIType.DUCKDB_TYPE_TIME);
        if (value == null) {
            return this.appendNull();
        }
        this.putLocalTime(col, this.rowIdx, value);
        this.moveToNextColumn();
        return this;
    }

    public DuckDBAppender appendDayMicros(long micros, int offset) throws SQLException {
        Column col = this.currentColumn(DuckDBBindings.CAPIType.DUCKDB_TYPE_TIME_TZ);
        this.putDayMicros(col, this.rowIdx, micros, offset);
        this.moveToNextColumn();
        return this;
    }

    public DuckDBAppender append(OffsetTime value) throws SQLException {
        Column col = this.currentColumn(DuckDBBindings.CAPIType.DUCKDB_TYPE_TIME_TZ);
        if (value == null) {
            return this.appendNull();
        }
        this.putOffsetTime(col, this.rowIdx, value);
        this.moveToNextColumn();
        return this;
    }

    public DuckDBAppender appendEpochSeconds(long seconds) throws SQLException {
        Column col = this.currentColumn(DuckDBBindings.CAPIType.DUCKDB_TYPE_TIMESTAMP_S);
        this.putEpochMoment(col, this.rowIdx, seconds);
        this.moveToNextColumn();
        return this;
    }

    public DuckDBAppender appendEpochMillis(long millis) throws SQLException {
        Column col = this.currentColumn(DuckDBBindings.CAPIType.DUCKDB_TYPE_TIMESTAMP_MS);
        this.putEpochMoment(col, this.rowIdx, millis);
        this.moveToNextColumn();
        return this;
    }

    public DuckDBAppender appendEpochMicros(long micros) throws SQLException {
        Column col = this.currentColumn(timestampMicrosTypes);
        this.putEpochMoment(col, this.rowIdx, micros);
        this.moveToNextColumn();
        return this;
    }

    public DuckDBAppender appendEpochNanos(long nanos) throws SQLException {
        Column col = this.currentColumn(DuckDBBindings.CAPIType.DUCKDB_TYPE_TIMESTAMP_NS);
        this.putEpochMoment(col, this.rowIdx, nanos);
        this.moveToNextColumn();
        return this;
    }

    public DuckDBAppender append(LocalDateTime value) throws SQLException {
        Column col = this.currentColumn(timestampLocalTypes);
        if (value == null) {
            return this.appendNull();
        }
        this.putLocalDateTime(col, this.rowIdx, value);
        this.moveToNextColumn();
        return this;
    }

    public DuckDBAppender append(Date value) throws SQLException {
        Column col = this.currentColumn(timestampLocalTypes);
        if (value == null) {
            return this.appendNull();
        }
        this.putDate(col, this.rowIdx, value);
        this.moveToNextColumn();
        return this;
    }

    public DuckDBAppender append(OffsetDateTime value) throws SQLException {
        Column col = this.currentColumn(DuckDBBindings.CAPIType.DUCKDB_TYPE_TIMESTAMP_TZ);
        if (value == null) {
            return this.appendNull();
        }
        this.putOffsetDateTime(col, this.rowIdx, value);
        this.moveToNextColumn();
        return this;
    }

    public DuckDBAppender append(Collection<?> collection) throws SQLException {
        this.currentColumn(collectionTypes);
        if (collection == null) {
            return this.appendNull();
        }
        return this.append(collection, collection.size());
    }

    public DuckDBAppender append(Iterable<?> iter, int count) throws SQLException {
        this.currentColumn(collectionTypes);
        if (iter == null) {
            return this.appendNull();
        }
        return this.append(iter.iterator(), count);
    }

    public DuckDBAppender append(Iterator<?> iter, int count) throws SQLException {
        Column parentCol = this.currentColumn(collectionTypes);
        if (parentCol.colType != DuckDBBindings.CAPIType.DUCKDB_TYPE_ARRAY && parentCol.colType != DuckDBBindings.CAPIType.DUCKDB_TYPE_LIST) {
            throw new SQLException(this.createErrMsg("invalid array/list column type: '" + (Object)((Object)parentCol.colType) + "'"));
        }
        if (iter == null) {
            return this.appendNull();
        }
        Column col = (Column)parentCol.children.get(0);
        this.putObjectArrayOrList(col, this.rowIdx, iter, count);
        this.moveToNextColumn();
        return this;
    }

    public DuckDBAppender append(Map<?, ?> map) throws SQLException {
        Column parentCol = this.currentColumn(DuckDBBindings.CAPIType.DUCKDB_TYPE_MAP);
        if (map == null) {
            return this.appendNull();
        }
        Column col = (Column)parentCol.children.get(0);
        this.putMap(col, this.rowIdx, map);
        this.moveToNextColumn();
        return this;
    }

    public DuckDBAppender appendNull() throws SQLException {
        Column col = this.currentColumn();
        col.setNull(this.rowIdx);
        this.moveToNextColumn();
        return this;
    }

    public DuckDBAppender appendDefault() throws SQLException {
        Column col = this.currentColumn();
        this.appenderRefLock.lock();
        try {
            this.checkOpen();
            DuckDBBindings.duckdb_append_default_to_chunk(this.appenderRef, this.chunkRef, col.idx, this.rowIdx);
        }
        finally {
            this.appenderRefLock.unlock();
        }
        this.moveToNextColumn();
        return this;
    }

    public boolean getWriteInlinedStrings() {
        return this.writeInlinedStrings;
    }

    public DuckDBAppender setWriteInlinedStrings(boolean writeInlinedStrings) {
        this.writeInlinedStrings = writeInlinedStrings;
        return this;
    }

    private String createErrMsg(String error) {
        return "Appender error, catalog: '" + this.catalog + "', schema: '" + this.schema + "', table: '" + this.table + "', message: " + (null != error ? error : "N/A");
    }

    private Column nextColumn(Column curCol) {
        if (null == curCol) {
            return null;
        }
        List cols = null == curCol.parent ? this.columns : curCol.parent.children;
        int nextColIdx = curCol.idx + 1;
        if (nextColIdx < cols.size()) {
            return (Column)cols.get(nextColIdx);
        }
        if (null != curCol.parent) {
            curCol = curCol.parent;
            return this.nextColumn(curCol);
        }
        return null;
    }

    private void moveToNextColumn() throws SQLException {
        Column col = this.currentColumn();
        this.prevColumn = this.currentColumn;
        this.currentColumn = this.unionBegunInvariant() ? this.nextColumn(col.parent) : this.nextColumn(col);
    }

    private void checkOpen() throws SQLException {
        if (this.isClosed()) {
            throw new SQLException(this.createErrMsg("appender was closed"));
        }
    }

    private void checkCurrentColumnType(DuckDBBindings.CAPIType ctype) throws SQLException {
        this.checkCurrentColumnType(ctype.typeArray);
    }

    private void checkCurrentColumnType(DuckDBBindings.CAPIType[] ctypes) throws SQLException {
        Column col = this.currentColumn();
        this.checkColumnType(col, ctypes);
    }

    private void checkColumnType(Column col, DuckDBBindings.CAPIType ctype) throws SQLException {
        this.checkColumnType(col, ctype.typeArray);
    }

    private void checkColumnType(Column col, DuckDBBindings.CAPIType[] ctypes) throws SQLException {
        for (DuckDBBindings.CAPIType ct : ctypes) {
            if (col.colType != ct) continue;
            return;
        }
        throw new SQLException(this.createErrMsg("invalid column type, expected one of: '" + Arrays.toString((Object[])ctypes) + "', actual: '" + (Object)((Object)col.colType) + "'"));
    }

    private void checkArrayLength(Column col, long length) throws SQLException {
        if (null == col.parent) {
            throw new SQLException(this.createErrMsg("invalid array/list column specified"));
        }
        switch (col.parent.colType) {
            case DUCKDB_TYPE_LIST: {
                return;
            }
            case DUCKDB_TYPE_ARRAY: {
                break;
            }
            default: {
                throw new SQLException(this.createErrMsg("invalid array/list column type: " + (Object)((Object)col.colType)));
            }
        }
        if (col.arraySize != length) {
            throw new SQLException(this.createErrMsg("invalid array size, expected: " + col.arraySize + ", actual: " + length));
        }
    }

    private void checkDecimalType(Column col, DuckDBBindings.CAPIType decimalInternalType) throws SQLException {
        if (col.decimalInternalType != decimalInternalType) {
            throw new SQLException(this.createErrMsg("invalid decimal internal type, expected: '" + (Object)((Object)col.decimalInternalType) + "', actual: '" + (Object)((Object)decimalInternalType) + "'"));
        }
    }

    private void checkDecimalPrecision(BigDecimal value, DuckDBBindings.CAPIType decimalInternalType, int maxPrecision) throws SQLException {
        if (value.precision() > maxPrecision) {
            throw new SQLException(this.createErrMsg("invalid decimal precision, value: " + value.precision() + ", max value: " + maxPrecision + ", decimal internal type: " + (Object)((Object)decimalInternalType)));
        }
    }

    private Column currentColumn() throws SQLException {
        this.checkOpen();
        if (null == this.currentColumn) {
            throw new SQLException(this.createErrMsg("current column not found, columns count: " + this.columns.size()));
        }
        return this.currentColumn;
    }

    private Column currentColumn(DuckDBBindings.CAPIType ctype) throws SQLException {
        return this.currentColumn(ctype.typeArray);
    }

    private Column currentColumn(DuckDBBindings.CAPIType[] ctypes) throws SQLException {
        Column col = this.currentColumn();
        this.checkColumnType(col, ctypes);
        return col;
    }

    private Column arrayInnerColumn(Column arrayCol, DuckDBBindings.CAPIType ctype) throws SQLException {
        return this.arrayInnerColumn(arrayCol, ctype.typeArray);
    }

    private Column arrayInnerColumn(Column arrayCol, DuckDBBindings.CAPIType[] ctypes) throws SQLException {
        if (arrayCol.colType != DuckDBBindings.CAPIType.DUCKDB_TYPE_ARRAY && arrayCol.colType != DuckDBBindings.CAPIType.DUCKDB_TYPE_LIST) {
            throw new SQLException(this.createErrMsg("invalid array/list column type: '" + (Object)((Object)arrayCol.colType) + "'"));
        }
        Column col = (Column)arrayCol.children.get(0);
        for (DuckDBBindings.CAPIType ct : ctypes) {
            if (col.colType != ct) continue;
            return col;
        }
        throw new SQLException(this.createErrMsg("invalid array/list inner column type, expected one of: '" + Arrays.toString((Object[])ctypes) + "', actual: '" + (Object)((Object)col.colType) + "'"));
    }

    private Column currentTopLevelColumn() {
        if (null == this.currentColumn) {
            return null;
        }
        Column col = this.currentColumn;
        while (null != col.parent) {
            col = col.parent;
        }
        return col;
    }

    private void setNullMask(Column col, long vectorIdx, boolean[] nullMask, int elementsCount) throws SQLException {
        if (null == col.parent) {
            throw new SQLException(this.createErrMsg("invalid array/list column specified"));
        }
        switch (col.parent.colType) {
            case DUCKDB_TYPE_ARRAY: {
                this.setArrayNullMask(col, vectorIdx, nullMask, elementsCount, 0);
                return;
            }
            case DUCKDB_TYPE_LIST: {
                this.setListNullMask(col, nullMask, elementsCount);
                return;
            }
        }
        throw new SQLException(this.createErrMsg("invalid array/list column type: " + (Object)((Object)col.colType)));
    }

    private void setArrayNullMask(Column col, long vectorIdx, boolean[] nullMask, int elementsCount, int parentArrayIdx) throws SQLException {
        if (null == nullMask) {
            return;
        }
        if (nullMask.length != elementsCount) {
            throw new SQLException(this.createErrMsg("invalid null mask size, expected: " + elementsCount + ", actual: " + nullMask.length));
        }
        for (int i = 0; i < nullMask.length; ++i) {
            if (!nullMask[i]) continue;
            col.setNullOnArrayIdx(vectorIdx, (int)((long)i + col.arraySize * (long)parentArrayIdx));
        }
    }

    private void setListNullMask(Column col, boolean[] nullMask, int elementsCount) throws SQLException {
        if (null == nullMask) {
            return;
        }
        if (nullMask.length != elementsCount) {
            throw new SQLException(this.createErrMsg("invalid null mask size, expected: " + elementsCount + ", actual: " + nullMask.length));
        }
        if (col.listSize < (long)elementsCount) {
            throw new SQLException(this.createErrMsg("invalid list state, list size: " + col.listSize + ", elements count: " + elementsCount));
        }
        for (int i = 0; i < nullMask.length; ++i) {
            if (!nullMask[i]) continue;
            long vectorIdx = col.listSize - (long)elementsCount + (long)i;
            col.setNull(vectorIdx);
        }
    }

    private void putByte(Column col, long vectorIdx, byte value) throws SQLException {
        int pos = (int)(vectorIdx * ((Column)col).colType.widthBytes);
        col.data.position(pos);
        col.data.put(value);
    }

    private void putShort(Column col, long vectorIdx, short value) throws SQLException {
        int pos = (int)(vectorIdx * ((Column)col).colType.widthBytes);
        col.data.position(pos);
        col.data.putShort(value);
    }

    private void putInt(Column col, long vectorIdx, int value) throws SQLException {
        int pos = (int)(vectorIdx * ((Column)col).colType.widthBytes);
        col.data.position(pos);
        col.data.putInt(value);
    }

    private void putLong(Column col, long vectorIdx, long value) throws SQLException {
        int pos = (int)(vectorIdx * ((Column)col).colType.widthBytes);
        col.data.position(pos);
        col.data.putLong(value);
    }

    private void putFloat(Column col, long vectorIdx, float value) throws SQLException {
        int pos = (int)(vectorIdx * ((Column)col).colType.widthBytes);
        col.data.position(pos);
        col.data.putFloat(value);
    }

    private void putDouble(Column col, long vectorIdx, double value) throws SQLException {
        int pos = (int)(vectorIdx * ((Column)col).colType.widthBytes);
        col.data.position(pos);
        col.data.putDouble(value);
    }

    private void putHugeInt(Column col, long vectorIdx, long lower, long upper) throws SQLException {
        int pos = (int)(vectorIdx * ((Column)col).colType.widthBytes);
        col.data.position(pos);
        col.data.putLong(lower);
        col.data.putLong(upper);
    }

    private void putBigInteger(Column col, long vectorIdx, BigInteger value) throws SQLException {
        if (value.compareTo(DuckDBHugeInt.HUGE_INT_MIN) < 0 || value.compareTo(DuckDBHugeInt.HUGE_INT_MAX) > 0) {
            throw new SQLException("Specified BigInteger value is out of range for HUGEINT field");
        }
        long lower = value.longValue();
        long upper = value.shiftRight(64).longValue();
        this.putHugeInt(col, vectorIdx, lower, upper);
    }

    private void putDecimal(Column col, long vectorIdx, short value) throws SQLException {
        int pos = (int)(vectorIdx * ((Column)col).decimalInternalType.widthBytes);
        col.data.position(pos);
        col.data.putShort(value);
    }

    private void putDecimal(Column col, long vectorIdx, int value) throws SQLException {
        int pos = (int)(vectorIdx * ((Column)col).decimalInternalType.widthBytes);
        col.data.position(pos);
        col.data.putInt(value);
    }

    private void putDecimal(Column col, long vectorIdx, long value) throws SQLException {
        int pos = (int)(vectorIdx * ((Column)col).decimalInternalType.widthBytes);
        col.data.position(pos);
        col.data.putLong(value);
    }

    private void putDecimal(Column col, long vectorIdx, long lower, long upper) throws SQLException {
        int pos = (int)(vectorIdx * ((Column)col).decimalInternalType.widthBytes);
        col.data.position(pos);
        col.data.putLong(lower);
        col.data.putLong(upper);
    }

    private void putDecimal(Column col, long vectorIdx, BigDecimal value) throws SQLException {
        if (value.precision() > col.decimalPrecision) {
            throw new SQLException(this.createErrMsg("invalid decimal precision, max expected: " + col.decimalPrecision + ", actual: " + value.precision()));
        }
        if (col.decimalScale != value.scale()) {
            throw new SQLException(this.createErrMsg("invalid decimal scale, expected: " + col.decimalScale + ", actual: " + value.scale()));
        }
        switch (col.decimalInternalType) {
            case DUCKDB_TYPE_SMALLINT: {
                this.checkDecimalPrecision(value, DuckDBBindings.CAPIType.DUCKDB_TYPE_SMALLINT, 4);
                short shortValue = value.unscaledValue().shortValueExact();
                this.putDecimal(col, vectorIdx, shortValue);
                break;
            }
            case DUCKDB_TYPE_INTEGER: {
                this.checkDecimalPrecision(value, DuckDBBindings.CAPIType.DUCKDB_TYPE_INTEGER, 9);
                int intValue = value.unscaledValue().intValueExact();
                this.putDecimal(col, vectorIdx, intValue);
                break;
            }
            case DUCKDB_TYPE_BIGINT: {
                this.checkDecimalPrecision(value, DuckDBBindings.CAPIType.DUCKDB_TYPE_BIGINT, 18);
                long longValue = value.unscaledValue().longValueExact();
                this.putDecimal(col, vectorIdx, longValue);
                break;
            }
            case DUCKDB_TYPE_HUGEINT: {
                this.checkDecimalPrecision(value, DuckDBBindings.CAPIType.DUCKDB_TYPE_HUGEINT, 38);
                BigInteger unscaledValue = value.unscaledValue();
                long lower = unscaledValue.longValue();
                long upper = unscaledValue.shiftRight(64).longValue();
                this.putDecimal(col, vectorIdx, lower, upper);
                break;
            }
            default: {
                throw new SQLException(this.createErrMsg("invalid decimal internal type: '" + (Object)((Object)col.decimalInternalType) + "'"));
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void putStringOrBlob(Column col, long vectorIdx, byte[] bytes) throws SQLException {
        if (this.writeInlinedStrings && bytes.length < 12) {
            int pos = (int)(vectorIdx * ((Column)col).colType.widthBytes);
            col.data.position(pos);
            col.data.putInt(bytes.length);
            if (bytes.length > 0) {
                col.data.put(bytes);
            }
        } else {
            this.appenderRefLock.lock();
            try {
                this.checkOpen();
                DuckDBBindings.duckdb_vector_assign_string_element_len(col.vectorRef, vectorIdx, bytes);
            }
            finally {
                this.appenderRefLock.unlock();
            }
        }
    }

    private void putUUID(Column col, long vectorIdx, long mostSigBits, long leastSigBits) throws SQLException {
        int pos = (int)(vectorIdx * ((Column)col).colType.widthBytes);
        col.data.position(pos);
        col.data.putLong(leastSigBits);
        col.data.putLong(mostSigBits ^= Long.MIN_VALUE);
    }

    private void putEpochDays(Column col, long vectorIdx, int days) throws SQLException {
        int pos = (int)(vectorIdx * ((Column)col).colType.widthBytes);
        col.data.position(pos);
        col.data.putInt(days);
    }

    private void putLocalDate(Column col, long vectorIdx, LocalDate date) throws SQLException {
        long days = date.toEpochDay();
        if (days < Integer.MIN_VALUE || days > Integer.MAX_VALUE) {
            throw new SQLException(this.createErrMsg("unsupported number of days: " + days + ", must fit into 'int32_t'"));
        }
        this.putEpochDays(col, vectorIdx, (int)days);
    }

    public void putDayMicros(Column col, long vectorIdx, long micros) throws SQLException {
        int pos = (int)(vectorIdx * ((Column)col).colType.widthBytes);
        col.data.position(pos);
        col.data.putLong(micros);
    }

    public void putLocalTime(Column col, long vectorIdx, LocalTime value) throws SQLException {
        long micros = value.toNanoOfDay() / 1000L;
        this.putDayMicros(col, vectorIdx, micros);
    }

    public void putDayMicros(Column col, long vectorIdx, long micros, int offset) throws SQLException {
        int maxOffset = 57599;
        long packed = (micros & 0xFFFFFFFFFFL) << 24 | (long)(maxOffset - offset & 0xFFFFFF);
        int pos = (int)(vectorIdx * ((Column)col).colType.widthBytes);
        col.data.position(pos);
        col.data.putLong(packed);
    }

    public void putOffsetTime(Column col, long vectorIdx, OffsetTime value) throws SQLException {
        int offset = value.getOffset().getTotalSeconds();
        long micros = value.toLocalTime().toNanoOfDay() / 1000L;
        this.putDayMicros(col, vectorIdx, micros, offset);
    }

    private void putEpochMoment(Column col, long vectorIdx, long moment) throws SQLException {
        int pos = (int)(vectorIdx * ((Column)col).colType.widthBytes);
        col.data.position(pos);
        col.data.putLong(moment);
    }

    private void putLocalDateTime(Column col, long vectorIdx, LocalDateTime value) throws SQLException {
        long moment;
        switch (col.colType) {
            case DUCKDB_TYPE_TIMESTAMP_S: {
                moment = EPOCH_DATE_TIME.until(value, ChronoUnit.SECONDS);
                break;
            }
            case DUCKDB_TYPE_TIMESTAMP_MS: {
                moment = EPOCH_DATE_TIME.until(value, ChronoUnit.MILLIS);
                break;
            }
            case DUCKDB_TYPE_TIMESTAMP: {
                moment = EPOCH_DATE_TIME.until(value, ChronoUnit.MICROS);
                break;
            }
            case DUCKDB_TYPE_TIMESTAMP_NS: {
                moment = EPOCH_DATE_TIME.until(value, ChronoUnit.NANOS);
                break;
            }
            default: {
                throw new SQLException(this.createErrMsg("invalid column type: " + (Object)((Object)col.colType)));
            }
        }
        this.putEpochMoment(col, vectorIdx, moment);
    }

    private void putDate(Column col, long vectorIdx, Date value) throws SQLException {
        long moment;
        switch (col.colType) {
            case DUCKDB_TYPE_TIMESTAMP_S: {
                moment = value.getTime() / 1000L;
                break;
            }
            case DUCKDB_TYPE_TIMESTAMP_MS: {
                moment = value.getTime();
                break;
            }
            case DUCKDB_TYPE_TIMESTAMP: {
                moment = Math.multiplyExact(value.getTime(), 1000L);
                break;
            }
            case DUCKDB_TYPE_TIMESTAMP_NS: {
                moment = Math.multiplyExact(value.getTime(), 1000000L);
                break;
            }
            default: {
                throw new SQLException(this.createErrMsg("invalid column type: " + (Object)((Object)col.colType)));
            }
        }
        this.putEpochMoment(col, vectorIdx, moment);
    }

    private void putOffsetDateTime(Column col, long vectorIdx, OffsetDateTime value) throws SQLException {
        ZonedDateTime zdt = value.atZoneSameInstant(ZoneOffset.UTC);
        LocalDateTime ldt = zdt.toLocalDateTime();
        long micros = EPOCH_DATE_TIME.until(ldt, ChronoUnit.MICROS);
        this.putEpochMoment(col, vectorIdx, micros);
    }

    private void putBoolArray(Column arrayCol, long vectorIdx, boolean[] values) throws SQLException {
        this.putBoolArray(arrayCol, vectorIdx, values, null);
    }

    private void putBoolArray(Column arrayCol, long vectorIdx, boolean[] values, boolean[] nullMask) throws SQLException {
        Column col = this.arrayInnerColumn(arrayCol, DuckDBBindings.CAPIType.DUCKDB_TYPE_BOOLEAN);
        byte[] bytes = new byte[values.length];
        for (int i = 0; i < values.length; ++i) {
            bytes[i] = (byte)(values[i] ? 1 : 0);
        }
        this.checkArrayLength(col, values.length);
        int pos = this.prepareListColumn(col, vectorIdx, values.length);
        this.setNullMask(col, vectorIdx, nullMask, values.length);
        col.data.position(pos);
        col.data.put(bytes);
    }

    private void putByteArray(Column arrayCol, long vectorIdx, byte[] values) throws SQLException {
        this.putByteArray(arrayCol, vectorIdx, values, null);
    }

    private void putByteArray(Column arrayCol, long vectorIdx, byte[] values, boolean[] nullMask) throws SQLException {
        Column col = this.arrayInnerColumn(arrayCol, int8Types);
        this.checkArrayLength(col, values.length);
        int pos = this.prepareListColumn(col, vectorIdx, values.length);
        this.setNullMask(col, vectorIdx, nullMask, values.length);
        col.data.position(pos);
        col.data.put(values);
    }

    private void putShortArray(Column arrayCol, long vectorIdx, short[] values) throws SQLException {
        this.putShortArray(arrayCol, vectorIdx, values, null);
    }

    private void putShortArray(Column arrayCol, long vectorIdx, short[] values, boolean[] nullMask) throws SQLException {
        Column col = this.arrayInnerColumn(arrayCol, int16Types);
        this.checkArrayLength(col, values.length);
        int pos = this.prepareListColumn(col, vectorIdx, values.length);
        this.setNullMask(col, vectorIdx, nullMask, values.length);
        ShortBuffer shortData = col.data.asShortBuffer();
        shortData.position(pos);
        shortData.put(values);
    }

    private void putIntArray(Column arrayCol, long vectorIdx, int[] values) throws SQLException {
        this.putIntArray(arrayCol, vectorIdx, values, null);
    }

    private void putIntArray(Column arrayCol, long vectorIdx, int[] values, boolean[] nullMask) throws SQLException {
        Column col = this.arrayInnerColumn(arrayCol, int32Types);
        this.checkArrayLength(col, values.length);
        int pos = this.prepareListColumn(col, vectorIdx, values.length);
        this.setNullMask(col, vectorIdx, nullMask, values.length);
        IntBuffer intData = col.data.asIntBuffer();
        intData.position(pos);
        intData.put(values);
    }

    private void putLongArray(Column arrayCol, long vectorIdx, long[] values) throws SQLException {
        this.putLongArray(arrayCol, vectorIdx, values, null);
    }

    private void putLongArray(Column arrayCol, long vectorIdx, long[] values, boolean[] nullMask) throws SQLException {
        Column col = this.arrayInnerColumn(arrayCol, int64Types);
        this.checkArrayLength(col, values.length);
        int pos = this.prepareListColumn(col, vectorIdx, values.length);
        this.setNullMask(col, vectorIdx, nullMask, values.length);
        LongBuffer longData = col.data.asLongBuffer();
        longData.position(pos);
        longData.put(values);
    }

    private void putFloatArray(Column arrayCol, long vectorIdx, float[] values) throws SQLException {
        this.putFloatArray(arrayCol, vectorIdx, values, null);
    }

    private void putFloatArray(Column arrayCol, long vectorIdx, float[] values, boolean[] nullMask) throws SQLException {
        Column col = this.arrayInnerColumn(arrayCol, DuckDBBindings.CAPIType.DUCKDB_TYPE_FLOAT);
        this.checkArrayLength(col, values.length);
        int pos = this.prepareListColumn(col, vectorIdx, values.length);
        this.setNullMask(col, vectorIdx, nullMask, values.length);
        FloatBuffer floatData = col.data.asFloatBuffer();
        floatData.position(pos);
        floatData.put(values);
    }

    private void putDoubleArray(Column arrayCol, long vectorIdx, double[] values) throws SQLException {
        this.putDoubleArray(arrayCol, vectorIdx, values, null);
    }

    private void putDoubleArray(Column arrayCol, long vectorIdx, double[] values, boolean[] nullMask) throws SQLException {
        Column col = this.arrayInnerColumn(arrayCol, DuckDBBindings.CAPIType.DUCKDB_TYPE_DOUBLE);
        this.checkArrayLength(col, values.length);
        int pos = this.prepareListColumn(col, vectorIdx, values.length);
        this.setNullMask(col, vectorIdx, nullMask, values.length);
        DoubleBuffer doubleData = col.data.asDoubleBuffer();
        doubleData.position(pos);
        doubleData.put(values);
    }

    private void putBoolArray2D(Column col, long vectorIdx, boolean[][] values) throws SQLException {
        this.putBoolArray2D(col, vectorIdx, values, null);
    }

    private void putBoolArray2D(Column outerCol, long vectorIdx, boolean[][] values, boolean[][] nullMask) throws SQLException {
        Column innerCol = this.arrayInnerColumn(outerCol, DuckDBBindings.CAPIType.DUCKDB_TYPE_ARRAY);
        this.checkArrayLength(innerCol, values.length);
        Column col = this.arrayInnerColumn(innerCol, DuckDBBindings.CAPIType.DUCKDB_TYPE_BOOLEAN);
        byte[] buf = new byte[(int)col.arraySize];
        for (int i = 0; i < values.length; ++i) {
            boolean[] childValues = values[i];
            if (childValues == null) {
                innerCol.setNullOnArrayIdx(vectorIdx, i);
                continue;
            }
            this.checkArrayLength(col, childValues.length);
            if (nullMask != null) {
                this.setArrayNullMask(col, vectorIdx, nullMask[i], childValues.length, i);
            }
            for (int j = 0; j < childValues.length; ++j) {
                buf[j] = (byte)(childValues[j] ? 1 : 0);
            }
            int pos = (int)((vectorIdx * innerCol.arraySize + (long)i) * col.arraySize);
            col.data.position(pos);
            col.data.put(buf);
        }
    }

    private void putByteArray2D(Column outerCol, long vectorIdx, byte[][] values) throws SQLException {
        this.putByteArray2D(outerCol, vectorIdx, values, null);
    }

    private void putByteArray2D(Column outerCol, long vectorIdx, byte[][] values, boolean[][] nullMask) throws SQLException {
        Column innerCol = this.arrayInnerColumn(outerCol, DuckDBBindings.CAPIType.DUCKDB_TYPE_ARRAY);
        this.checkArrayLength(innerCol, values.length);
        Column col = this.arrayInnerColumn(innerCol, int8Types);
        for (int i = 0; i < values.length; ++i) {
            byte[] childValues = values[i];
            if (childValues == null) {
                innerCol.setNullOnArrayIdx(vectorIdx, i);
                continue;
            }
            this.checkArrayLength(col, childValues.length);
            if (nullMask != null) {
                this.setArrayNullMask(col, vectorIdx, nullMask[i], childValues.length, i);
            }
            int pos = (int)((vectorIdx * innerCol.arraySize + (long)i) * col.arraySize);
            col.data.position(pos);
            col.data.put(childValues);
        }
    }

    private void putShortArray2D(Column outerCol, long vectorIdx, short[][] values) throws SQLException {
        this.putShortArray2D(outerCol, vectorIdx, values, null);
    }

    private void putShortArray2D(Column outerCol, long vectorIdx, short[][] values, boolean[][] nullMask) throws SQLException {
        Column innerCol = this.arrayInnerColumn(outerCol, DuckDBBindings.CAPIType.DUCKDB_TYPE_ARRAY);
        this.checkArrayLength(innerCol, values.length);
        Column col = this.arrayInnerColumn(innerCol, int16Types);
        for (int i = 0; i < values.length; ++i) {
            short[] childValues = values[i];
            if (childValues == null) {
                innerCol.setNullOnArrayIdx(vectorIdx, i);
                continue;
            }
            this.checkArrayLength(col, childValues.length);
            if (nullMask != null) {
                this.setArrayNullMask(col, vectorIdx, nullMask[i], childValues.length, i);
            }
            ShortBuffer shortBuffer = col.data.asShortBuffer();
            int pos = (int)((vectorIdx * innerCol.arraySize + (long)i) * col.arraySize);
            shortBuffer.position(pos);
            shortBuffer.put(childValues);
        }
    }

    private void putIntArray2D(Column col, long vectorIdx, int[][] values) throws SQLException {
        this.putIntArray2D(col, vectorIdx, values, null);
    }

    private void putIntArray2D(Column outerCol, long vectorIdx, int[][] values, boolean[][] nullMask) throws SQLException {
        Column innerCol = this.arrayInnerColumn(outerCol, DuckDBBindings.CAPIType.DUCKDB_TYPE_ARRAY);
        this.checkArrayLength(innerCol, values.length);
        Column col = this.arrayInnerColumn(innerCol, int32Types);
        for (int i = 0; i < values.length; ++i) {
            int[] childValues = values[i];
            if (childValues == null) {
                innerCol.setNullOnArrayIdx(this.rowIdx, i);
                continue;
            }
            this.checkArrayLength(col, childValues.length);
            if (nullMask != null) {
                this.setArrayNullMask(col, vectorIdx, nullMask[i], childValues.length, i);
            }
            IntBuffer intData = col.data.asIntBuffer();
            int pos = (int)((vectorIdx * innerCol.arraySize + (long)i) * col.arraySize);
            intData.position(pos);
            intData.put(childValues);
        }
    }

    private void putLongArray2D(Column outerCol, long vectorIdx, long[][] values) throws SQLException {
        this.putLongArray2D(outerCol, vectorIdx, values, null);
    }

    private void putLongArray2D(Column outerCol, long vectorIdx, long[][] values, boolean[][] nullMask) throws SQLException {
        Column innerCol = this.arrayInnerColumn(outerCol, DuckDBBindings.CAPIType.DUCKDB_TYPE_ARRAY);
        this.checkArrayLength(innerCol, values.length);
        Column col = this.arrayInnerColumn(innerCol, int64Types);
        for (int i = 0; i < values.length; ++i) {
            long[] childValues = values[i];
            if (childValues == null) {
                innerCol.setNullOnArrayIdx(vectorIdx, i);
                continue;
            }
            this.checkArrayLength(col, childValues.length);
            if (nullMask != null) {
                this.setArrayNullMask(col, vectorIdx, nullMask[i], childValues.length, i);
            }
            LongBuffer longData = col.data.asLongBuffer();
            int pos = (int)((vectorIdx * innerCol.arraySize + (long)i) * col.arraySize);
            longData.position(pos);
            longData.put(childValues);
        }
    }

    private void putFloatArray2D(Column outerCol, long vectorIdx, float[][] values) throws SQLException {
        this.putFloatArray2D(outerCol, vectorIdx, values, null);
    }

    private void putFloatArray2D(Column outerCol, long vectorIdx, float[][] values, boolean[][] nullMask) throws SQLException {
        Column innerCol = this.arrayInnerColumn(outerCol, DuckDBBindings.CAPIType.DUCKDB_TYPE_ARRAY);
        this.checkArrayLength(innerCol, values.length);
        Column col = this.arrayInnerColumn(innerCol, DuckDBBindings.CAPIType.DUCKDB_TYPE_FLOAT);
        for (int i = 0; i < values.length; ++i) {
            float[] childValues = values[i];
            if (childValues == null) {
                innerCol.setNullOnArrayIdx(vectorIdx, i);
                continue;
            }
            this.checkArrayLength(col, childValues.length);
            if (nullMask != null) {
                this.setArrayNullMask(col, vectorIdx, nullMask[i], childValues.length, i);
            }
            FloatBuffer floatData = col.data.asFloatBuffer();
            int pos = (int)((vectorIdx * innerCol.arraySize + (long)i) * col.arraySize);
            floatData.position(pos);
            floatData.put(childValues);
        }
    }

    private void putDoubleArray2D(Column outerCol, long vectorIdx, double[][] values) throws SQLException {
        this.putDoubleArray2D(outerCol, vectorIdx, values, null);
    }

    private void putDoubleArray2D(Column outerCol, long vectorIdx, double[][] values, boolean[][] nullMask) throws SQLException {
        Column innerCol = this.arrayInnerColumn(outerCol, DuckDBBindings.CAPIType.DUCKDB_TYPE_ARRAY);
        this.checkArrayLength(innerCol, values.length);
        Column col = this.arrayInnerColumn(innerCol, DuckDBBindings.CAPIType.DUCKDB_TYPE_DOUBLE);
        for (int i = 0; i < values.length; ++i) {
            double[] childValues = values[i];
            if (childValues == null) {
                innerCol.setNullOnArrayIdx(vectorIdx, i);
                continue;
            }
            this.checkArrayLength(col, childValues.length);
            if (nullMask != null) {
                this.setArrayNullMask(col, vectorIdx, nullMask[i], childValues.length, i);
            }
            DoubleBuffer doubleBuffer = col.data.asDoubleBuffer();
            int pos = (int)((vectorIdx * innerCol.arraySize + (long)i) * col.arraySize);
            doubleBuffer.position(pos);
            doubleBuffer.put(childValues);
        }
    }

    private void putObjectArrayOrList(Column col, long vectorIdx, Iterator<?> iter, int count) throws SQLException {
        this.checkArrayLength(col, count);
        this.prepareListColumn(col, vectorIdx, count);
        long offset = col.parent.colType == DuckDBBindings.CAPIType.DUCKDB_TYPE_LIST ? col.listSize - (long)count : vectorIdx * col.arraySize;
        for (long i = 0L; i < (long)count; ++i) {
            if (!iter.hasNext()) {
                throw new SQLException(this.createErrMsg("invalid iterator elements count, expected: " + count + ", actual" + i));
            }
            Object value = iter.next();
            long innerVectorIdx = offset + i;
            this.putCompositeElement(col, innerVectorIdx, value);
        }
    }

    private void putMap(Column col, long vectorIdx, Map<?, ?> map) throws SQLException {
        this.prepareListColumn(col, vectorIdx, map.size());
        long offset = col.listSize - (long)map.size();
        long i = 0L;
        ArrayList values = new ArrayList(2);
        values.add(null);
        values.add(null);
        for (Map.Entry<?, ?> en : map.entrySet()) {
            values.set(0, en.getKey());
            values.set(1, en.getValue());
            long innerVectorIdx = offset + i;
            this.putCompositeElementStruct(col, innerVectorIdx, values);
            ++i;
        }
    }

    private void putCompositeElement(Column col, long vectorIdx, Object value) throws SQLException {
        if (null == value) {
            col.setNull(vectorIdx);
            return;
        }
        switch (col.colType) {
            case DUCKDB_TYPE_BOOLEAN: {
                Boolean bool = (Boolean)value;
                byte num = (byte)(bool != false ? 1 : 0);
                this.putByte(col, vectorIdx, num);
                break;
            }
            case DUCKDB_TYPE_TINYINT: {
                Byte num = (Byte)value;
                this.putByte(col, vectorIdx, num);
                break;
            }
            case DUCKDB_TYPE_SMALLINT: {
                Short num = (Short)value;
                this.putShort(col, vectorIdx, num);
                break;
            }
            case DUCKDB_TYPE_INTEGER: {
                Integer num = (Integer)value;
                this.putInt(col, vectorIdx, num);
                break;
            }
            case DUCKDB_TYPE_BIGINT: {
                Long num = (Long)value;
                this.putLong(col, vectorIdx, num);
                break;
            }
            case DUCKDB_TYPE_HUGEINT: {
                BigInteger num = (BigInteger)value;
                this.putBigInteger(col, vectorIdx, num);
                break;
            }
            case DUCKDB_TYPE_FLOAT: {
                Float num = (Float)value;
                this.putFloat(col, vectorIdx, num.floatValue());
                break;
            }
            case DUCKDB_TYPE_DOUBLE: {
                Double num = (Double)value;
                this.putDouble(col, vectorIdx, num);
                break;
            }
            case DUCKDB_TYPE_DECIMAL: {
                BigDecimal num = (BigDecimal)value;
                this.putDecimal(col, vectorIdx, num);
                break;
            }
            case DUCKDB_TYPE_VARCHAR: {
                String st = (String)value;
                byte[] bytes = DuckDBAppender.utf8(st);
                this.putStringOrBlob(col, vectorIdx, bytes);
                break;
            }
            case DUCKDB_TYPE_UUID: {
                UUID uid = (UUID)value;
                long mostSigBits = uid.getMostSignificantBits();
                long leastSigBits = uid.getLeastSignificantBits();
                this.putUUID(col, vectorIdx, mostSigBits, leastSigBits);
                break;
            }
            case DUCKDB_TYPE_DATE: {
                LocalDate ld = (LocalDate)value;
                this.putLocalDate(col, vectorIdx, ld);
                break;
            }
            case DUCKDB_TYPE_TIME: {
                LocalTime lt = (LocalTime)value;
                this.putLocalTime(col, vectorIdx, lt);
                break;
            }
            case DUCKDB_TYPE_TIME_TZ: {
                OffsetTime ot = (OffsetTime)value;
                this.putOffsetTime(col, vectorIdx, ot);
                break;
            }
            case DUCKDB_TYPE_TIMESTAMP_S: 
            case DUCKDB_TYPE_TIMESTAMP_MS: 
            case DUCKDB_TYPE_TIMESTAMP: 
            case DUCKDB_TYPE_TIMESTAMP_NS: {
                if (value instanceof LocalDateTime) {
                    LocalDateTime ldt = (LocalDateTime)value;
                    this.putLocalDateTime(col, vectorIdx, ldt);
                    break;
                }
                if (value instanceof Date) {
                    Date dt = (Date)value;
                    this.putDate(col, vectorIdx, dt);
                    break;
                }
                throw new SQLException(this.createErrMsg("invalid object type for timestamp column, expected one of: [" + LocalDateTime.class.getName() + ", " + Date.class.getName() + "], actual: [" + value.getClass().getName() + "]"));
            }
            case DUCKDB_TYPE_TIMESTAMP_TZ: {
                OffsetDateTime odt = (OffsetDateTime)value;
                this.putOffsetDateTime(col, vectorIdx, odt);
                break;
            }
            case DUCKDB_TYPE_ARRAY: {
                this.putCompositeElementArray(col, vectorIdx, value);
                break;
            }
            case DUCKDB_TYPE_LIST: {
                Collection collection = (Collection)value;
                if (col.children.size() != 1) {
                    throw new SQLException(this.createErrMsg("invalid list column"));
                }
                Column innerCol = (Column)col.children.get(0);
                this.putObjectArrayOrList(innerCol, vectorIdx, collection.iterator(), collection.size());
                break;
            }
            case DUCKDB_TYPE_MAP: {
                Map map = (Map)value;
                if (col.children.size() != 1) {
                    throw new SQLException(this.createErrMsg("invalid map column"));
                }
                Column innerCol = (Column)col.children.get(0);
                this.putMap(innerCol, vectorIdx, map);
                break;
            }
            case DUCKDB_TYPE_STRUCT: {
                this.putCompositeElementStruct(col, vectorIdx, value);
                break;
            }
            case DUCKDB_TYPE_UNION: {
                this.putCompositeElementUnion(col, vectorIdx, value);
                break;
            }
            default: {
                throw new SQLException(this.createErrMsg("unsupported composite column, inner type: " + (Object)((Object)col.colType)));
            }
        }
    }

    private void putCompositeElementArray(Column col, long vectorIdx, Object value) throws SQLException {
        if (col.children.size() != 1) {
            throw new SQLException(this.createErrMsg("invalid array column"));
        }
        Column innerCol = (Column)col.children.get(0);
        switch (innerCol.colType) {
            case DUCKDB_TYPE_BOOLEAN: {
                boolean[] arr = (boolean[])value;
                this.putBoolArray(col, vectorIdx, arr);
                break;
            }
            case DUCKDB_TYPE_TINYINT: {
                byte[] arr = (byte[])value;
                this.putByteArray(col, vectorIdx, arr);
                break;
            }
            case DUCKDB_TYPE_SMALLINT: {
                short[] arr = (short[])value;
                this.putShortArray(col, vectorIdx, arr);
                break;
            }
            case DUCKDB_TYPE_INTEGER: {
                int[] arr = (int[])value;
                this.putIntArray(col, vectorIdx, arr);
                break;
            }
            case DUCKDB_TYPE_BIGINT: {
                long[] arr = (long[])value;
                this.putLongArray(col, vectorIdx, arr);
                break;
            }
            case DUCKDB_TYPE_FLOAT: {
                float[] arr = (float[])value;
                this.putFloatArray(col, vectorIdx, arr);
                break;
            }
            case DUCKDB_TYPE_DOUBLE: {
                double[] arr = (double[])value;
                this.putDoubleArray(col, vectorIdx, arr);
                break;
            }
            case DUCKDB_TYPE_ARRAY: {
                if (value instanceof boolean[][]) {
                    boolean[][] arr = (boolean[][])value;
                    this.putBoolArray2D(col, vectorIdx, arr);
                    break;
                }
                if (value instanceof byte[][]) {
                    byte[][] arr = (byte[][])value;
                    this.putByteArray2D(col, vectorIdx, arr);
                    break;
                }
                if (value instanceof short[][]) {
                    short[][] arr = (short[][])value;
                    this.putShortArray2D(col, vectorIdx, arr);
                    break;
                }
                if (value instanceof int[][]) {
                    int[][] arr = (int[][])value;
                    this.putIntArray2D(col, vectorIdx, arr);
                    break;
                }
                if (value instanceof long[][]) {
                    long[][] arr = (long[][])value;
                    this.putLongArray2D(col, vectorIdx, arr);
                    break;
                }
                if (value instanceof float[][]) {
                    float[][] arr = (float[][])value;
                    this.putFloatArray2D(col, vectorIdx, arr);
                    break;
                }
                if (value instanceof double[][]) {
                    double[][] arr = (double[][])value;
                    this.putDoubleArray2D(col, vectorIdx, arr);
                    break;
                }
                throw new SQLException(this.createErrMsg("unsupported 2D array type: " + value.getClass().getName()));
            }
            default: {
                throw new SQLException(this.createErrMsg("unsupported array type: " + (Object)((Object)innerCol.colType)));
            }
        }
    }

    /*
     * Enabled force condition propagation
     * Lifted jumps to return sites
     */
    private void putCompositeElementStruct(Column structCol, long vectorIdx, Object structValue) throws SQLException {
        Collection collection;
        if (structValue instanceof Map) {
            if (!(structValue instanceof LinkedHashMap)) throw new SQLException(this.createErrMsg("struct values must be specified as an instance of a 'java.util.LinkedHashMap' or as a collection of objects, actual class: " + structValue.getClass().getName()));
            LinkedHashMap map = (LinkedHashMap)structValue;
            collection = map.values();
        } else {
            collection = (Collection)structValue;
        }
        if (structCol.children.size() != collection.size()) {
            throw new SQLException(this.createErrMsg("invalid struct object specified, expected fields count: " + structCol.children.size() + ", actual: " + collection.size()));
        }
        int i = 0;
        for (Object value : collection) {
            Column col = (Column)structCol.children.get(i);
            this.putCompositeElement(col, vectorIdx, value);
            ++i;
        }
    }

    private void putCompositeElementUnion(Column unionCol, long vectorIdx, Object unionValue) throws SQLException {
        if (!(unionValue instanceof AbstractMap.SimpleEntry)) {
            throw new SQLException(this.createErrMsg("union values must be specified as an instance of 'java.util.AbstractMap.SimpleEntry<String, Object>', actual type: " + unionValue.getClass().getName()));
        }
        AbstractMap.SimpleEntry entry = (AbstractMap.SimpleEntry)unionValue;
        String tag = String.valueOf(entry.getKey());
        Column col = this.putUnionTag(unionCol, vectorIdx, tag);
        this.putCompositeElement(col, vectorIdx, entry.getValue());
    }

    private Column putUnionTag(Column col, long vectorIdx, String tag) throws SQLException {
        int fieldWithTag = 0;
        for (int i = 1; i < col.children.size(); ++i) {
            Column childCol = (Column)col.children.get(i);
            if (!childCol.structFieldName.equals(tag)) continue;
            fieldWithTag = i;
        }
        if (0 == fieldWithTag) {
            throw new SQLException(this.createErrMsg("specified union field not found, value: '" + tag + "'"));
        }
        Column tagCol = (Column)col.children.get(0);
        this.putByte(tagCol, vectorIdx, (byte)(fieldWithTag - 1));
        for (int i = 1; i < col.children.size(); ++i) {
            if (i == fieldWithTag) continue;
            Column childCol = (Column)col.children.get(i);
            childCol.setNull(vectorIdx);
        }
        return (Column)col.children.get(fieldWithTag);
    }

    private boolean rowBegunInvariant() {
        return null != this.currentColumn;
    }

    private boolean rowCompletedInvariant() {
        return null == this.currentColumn && null != this.prevColumn && this.prevColumn.idx == this.columns.size() - 1;
    }

    private boolean structBegunInvariant() {
        return null != this.currentColumn && null != this.currentColumn.parent && this.currentColumn.parent.colType == DuckDBBindings.CAPIType.DUCKDB_TYPE_STRUCT;
    }

    private boolean structCompletedInvariant() {
        return null != this.prevColumn && null != this.prevColumn.parent && this.prevColumn.parent.colType == DuckDBBindings.CAPIType.DUCKDB_TYPE_STRUCT && this.prevColumn.idx == this.prevColumn.parent.children.size() - 1;
    }

    private boolean unionBegunInvariant() {
        return null != this.currentColumn && null != this.currentColumn.parent && this.currentColumn.parent.colType == DuckDBBindings.CAPIType.DUCKDB_TYPE_UNION;
    }

    private boolean unionCompletedInvariant() {
        return null != this.prevColumn && null != this.prevColumn.parent && this.prevColumn.parent.colType == DuckDBBindings.CAPIType.DUCKDB_TYPE_UNION;
    }

    private boolean readyForANewRowInvariant() {
        return null == this.currentColumn && null == this.prevColumn;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private int prepareListColumn(Column innerCol, long vectorIdx, long listElementsCount) throws SQLException {
        if (null == innerCol.parent) {
            throw new SQLException(this.createErrMsg("invalid array/list column specified"));
        }
        Column col = innerCol.parent;
        switch (col.colType) {
            case DUCKDB_TYPE_ARRAY: {
                return (int)(vectorIdx * innerCol.arraySize);
            }
            case DUCKDB_TYPE_LIST: 
            case DUCKDB_TYPE_MAP: {
                break;
            }
            default: {
                throw new SQLException(this.createErrMsg("invalid array/list column type: " + (Object)((Object)col.colType)));
            }
        }
        this.appenderRefLock.lock();
        try {
            this.checkOpen();
            long offset = DuckDBBindings.duckdb_list_vector_get_size(col.vectorRef);
            LongBuffer longBuffer = col.data.asLongBuffer();
            int pos = (int)(vectorIdx * DuckDBBindings.CAPIType.DUCKDB_TYPE_LIST.widthBytes / 8L);
            longBuffer.position(pos);
            longBuffer.put(offset);
            longBuffer.put(listElementsCount);
            long listSize = offset + listElementsCount;
            int reserveStatus = DuckDBBindings.duckdb_list_vector_reserve(col.vectorRef, listSize);
            if (0 != reserveStatus) {
                throw new SQLException(this.createErrMsg("'duckdb_list_vector_reserve' call failed, list size: " + listSize));
            }
            innerCol.reset(listSize);
            int setStatus = DuckDBBindings.duckdb_list_vector_set_size(col.vectorRef, listSize);
            if (0 != setStatus) {
                throw new SQLException(this.createErrMsg("'duckdb_list_vector_set_size' call failed, list size: " + listSize));
            }
            int n = (int)offset;
            return n;
        }
        finally {
            this.appenderRefLock.unlock();
        }
    }

    private static byte[] utf8(String str) {
        if (null == str) {
            return null;
        }
        return str.getBytes(StandardCharsets.UTF_8);
    }

    private static String strFromUTF8(byte[] utf8) {
        if (null == utf8) {
            return "";
        }
        return new String(utf8, StandardCharsets.UTF_8);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private static ByteBuffer createAppender(DuckDBConnection conn, String catalog, String schema, String table) throws SQLException {
        conn.checkOpen();
        ReentrantLock connRefLock = conn.connRefLock;
        connRefLock.lock();
        try {
            ByteBuffer[] out = new ByteBuffer[1];
            int state = DuckDBBindings.duckdb_appender_create_ext(conn.connRef, DuckDBAppender.utf8(catalog), DuckDBAppender.utf8(schema), DuckDBAppender.utf8(table), out);
            if (0 != state) {
                throw new SQLException("duckdb_appender_create_ext error");
            }
            ByteBuffer byteBuffer = out[0];
            return byteBuffer;
        }
        finally {
            connRefLock.unlock();
        }
    }

    private static ByteBuffer[] readTableTypes(ByteBuffer appenderRef) throws SQLException {
        long colCountLong = DuckDBBindings.duckdb_appender_column_count(appenderRef);
        if (colCountLong > Integer.MAX_VALUE || colCountLong < 0L) {
            throw new SQLException("invalid columns count: " + colCountLong);
        }
        int colCount = (int)colCountLong;
        ByteBuffer[] res = new ByteBuffer[colCount];
        for (int i = 0; i < colCount; ++i) {
            ByteBuffer colType = DuckDBBindings.duckdb_appender_column_type(appenderRef, i);
            if (null == colType) {
                throw new SQLException("cannot get logical type for column: " + i);
            }
            int typeId = DuckDBBindings.duckdb_get_type_id(colType);
            if (!supportedTypes.contains(typeId)) {
                for (ByteBuffer lt : res) {
                    if (null == lt) continue;
                    DuckDBBindings.duckdb_destroy_logical_type(lt);
                }
                throw new SQLException("unsupported C API type: " + typeId);
            }
            res[i] = colType;
        }
        return res;
    }

    private static ByteBuffer createChunk(ByteBuffer[] colTypes) throws SQLException {
        ByteBuffer chunkRef = DuckDBBindings.duckdb_create_data_chunk(colTypes);
        if (null == chunkRef) {
            throw new SQLException("cannot create data chunk");
        }
        return chunkRef;
    }

    private static void initVecChildren(Column parent) throws SQLException {
        switch (parent.colType) {
            case DUCKDB_TYPE_LIST: 
            case DUCKDB_TYPE_MAP: {
                ByteBuffer vec = DuckDBBindings.duckdb_list_vector_get_child(parent.vectorRef);
                Column col = new Column(parent, 0, null, vec);
                parent.children.add(col);
                break;
            }
            case DUCKDB_TYPE_STRUCT: 
            case DUCKDB_TYPE_UNION: {
                long count = DuckDBBindings.duckdb_struct_type_child_count(parent.colTypeRef);
                int i = 0;
                while ((long)i < count) {
                    ByteBuffer vec = DuckDBBindings.duckdb_struct_vector_get_child(parent.vectorRef, i);
                    Column col = new Column(parent, i, null, vec, i);
                    parent.children.add(col);
                    ++i;
                }
                break;
            }
            case DUCKDB_TYPE_ARRAY: {
                ByteBuffer vec = DuckDBBindings.duckdb_array_vector_get_child(parent.vectorRef);
                Column col = new Column(parent, 0, null, vec);
                parent.children.add(col);
                break;
            }
        }
    }

    private static List<Column> createTopLevelColumns(ByteBuffer chunkRef, ByteBuffer[] colTypes) throws SQLException {
        ArrayList<Column> columns = new ArrayList<Column>(colTypes.length);
        try {
            for (int i = 0; i < colTypes.length; ++i) {
                ByteBuffer vector = DuckDBBindings.duckdb_data_chunk_get_vector(chunkRef, i);
                Column col = new Column(null, i, colTypes[i], vector);
                columns.add(col);
                colTypes[i] = null;
            }
        }
        catch (Exception e) {
            for (Column col : columns) {
                if (null == col) continue;
                col.destroy();
            }
            throw e;
        }
        return columns;
    }

    static {
        supportedTypes.add(DuckDBBindings.CAPIType.DUCKDB_TYPE_BOOLEAN.typeId);
        supportedTypes.add(DuckDBBindings.CAPIType.DUCKDB_TYPE_TINYINT.typeId);
        supportedTypes.add(DuckDBBindings.CAPIType.DUCKDB_TYPE_UTINYINT.typeId);
        supportedTypes.add(DuckDBBindings.CAPIType.DUCKDB_TYPE_SMALLINT.typeId);
        supportedTypes.add(DuckDBBindings.CAPIType.DUCKDB_TYPE_USMALLINT.typeId);
        supportedTypes.add(DuckDBBindings.CAPIType.DUCKDB_TYPE_INTEGER.typeId);
        supportedTypes.add(DuckDBBindings.CAPIType.DUCKDB_TYPE_UINTEGER.typeId);
        supportedTypes.add(DuckDBBindings.CAPIType.DUCKDB_TYPE_BIGINT.typeId);
        supportedTypes.add(DuckDBBindings.CAPIType.DUCKDB_TYPE_UBIGINT.typeId);
        supportedTypes.add(DuckDBBindings.CAPIType.DUCKDB_TYPE_HUGEINT.typeId);
        supportedTypes.add(DuckDBBindings.CAPIType.DUCKDB_TYPE_UHUGEINT.typeId);
        supportedTypes.add(DuckDBBindings.CAPIType.DUCKDB_TYPE_FLOAT.typeId);
        supportedTypes.add(DuckDBBindings.CAPIType.DUCKDB_TYPE_DOUBLE.typeId);
        supportedTypes.add(DuckDBBindings.CAPIType.DUCKDB_TYPE_DECIMAL.typeId);
        supportedTypes.add(DuckDBBindings.CAPIType.DUCKDB_TYPE_VARCHAR.typeId);
        supportedTypes.add(DuckDBBindings.CAPIType.DUCKDB_TYPE_BLOB.typeId);
        supportedTypes.add(DuckDBBindings.CAPIType.DUCKDB_TYPE_DATE.typeId);
        supportedTypes.add(DuckDBBindings.CAPIType.DUCKDB_TYPE_TIME.typeId);
        supportedTypes.add(DuckDBBindings.CAPIType.DUCKDB_TYPE_TIME_TZ.typeId);
        supportedTypes.add(DuckDBBindings.CAPIType.DUCKDB_TYPE_TIMESTAMP_S.typeId);
        supportedTypes.add(DuckDBBindings.CAPIType.DUCKDB_TYPE_TIMESTAMP_MS.typeId);
        supportedTypes.add(DuckDBBindings.CAPIType.DUCKDB_TYPE_TIMESTAMP.typeId);
        supportedTypes.add(DuckDBBindings.CAPIType.DUCKDB_TYPE_TIMESTAMP_TZ.typeId);
        supportedTypes.add(DuckDBBindings.CAPIType.DUCKDB_TYPE_TIMESTAMP_NS.typeId);
        supportedTypes.add(DuckDBBindings.CAPIType.DUCKDB_TYPE_UUID.typeId);
        supportedTypes.add(DuckDBBindings.CAPIType.DUCKDB_TYPE_ARRAY.typeId);
        supportedTypes.add(DuckDBBindings.CAPIType.DUCKDB_TYPE_LIST.typeId);
        supportedTypes.add(DuckDBBindings.CAPIType.DUCKDB_TYPE_MAP.typeId);
        supportedTypes.add(DuckDBBindings.CAPIType.DUCKDB_TYPE_STRUCT.typeId);
        supportedTypes.add(DuckDBBindings.CAPIType.DUCKDB_TYPE_UNION.typeId);
        int8Types = new DuckDBBindings.CAPIType[]{DuckDBBindings.CAPIType.DUCKDB_TYPE_TINYINT, DuckDBBindings.CAPIType.DUCKDB_TYPE_UTINYINT};
        int16Types = new DuckDBBindings.CAPIType[]{DuckDBBindings.CAPIType.DUCKDB_TYPE_SMALLINT, DuckDBBindings.CAPIType.DUCKDB_TYPE_USMALLINT};
        int32Types = new DuckDBBindings.CAPIType[]{DuckDBBindings.CAPIType.DUCKDB_TYPE_INTEGER, DuckDBBindings.CAPIType.DUCKDB_TYPE_UINTEGER};
        int64Types = new DuckDBBindings.CAPIType[]{DuckDBBindings.CAPIType.DUCKDB_TYPE_BIGINT, DuckDBBindings.CAPIType.DUCKDB_TYPE_UBIGINT};
        int128Types = new DuckDBBindings.CAPIType[]{DuckDBBindings.CAPIType.DUCKDB_TYPE_HUGEINT, DuckDBBindings.CAPIType.DUCKDB_TYPE_UHUGEINT};
        timestampLocalTypes = new DuckDBBindings.CAPIType[]{DuckDBBindings.CAPIType.DUCKDB_TYPE_TIMESTAMP_S, DuckDBBindings.CAPIType.DUCKDB_TYPE_TIMESTAMP_MS, DuckDBBindings.CAPIType.DUCKDB_TYPE_TIMESTAMP, DuckDBBindings.CAPIType.DUCKDB_TYPE_TIMESTAMP_NS};
        timestampMicrosTypes = new DuckDBBindings.CAPIType[]{DuckDBBindings.CAPIType.DUCKDB_TYPE_TIMESTAMP, DuckDBBindings.CAPIType.DUCKDB_TYPE_TIMESTAMP_TZ};
        collectionTypes = new DuckDBBindings.CAPIType[]{DuckDBBindings.CAPIType.DUCKDB_TYPE_ARRAY, DuckDBBindings.CAPIType.DUCKDB_TYPE_LIST};
        EPOCH_DATE_TIME = LocalDateTime.ofEpochSecond(0L, 0, ZoneOffset.UTC);
    }

    private static class Column {
        private final Column parent;
        private final int idx;
        private ByteBuffer colTypeRef;
        private final DuckDBBindings.CAPIType colType;
        private final DuckDBBindings.CAPIType decimalInternalType;
        private final int decimalPrecision;
        private final int decimalScale;
        private final long arraySize;
        private final String structFieldName;
        private final ByteBuffer vectorRef;
        private final List<Column> children = new ArrayList<Column>();
        private long listSize = 0L;
        private ByteBuffer data = null;
        private ByteBuffer validity = null;

        private Column(Column parent, int idx, ByteBuffer colTypeRef, ByteBuffer vector) throws SQLException {
            this(parent, idx, colTypeRef, vector, -1);
        }

        private Column(Column parent, int idx, ByteBuffer colTypeRef, ByteBuffer vector, int structFieldIdx) throws SQLException {
            this.parent = parent;
            this.idx = idx;
            if (null == vector) {
                throw new SQLException("cannot initialize data chunk vector");
            }
            if (null == colTypeRef) {
                this.colTypeRef = DuckDBBindings.duckdb_vector_get_column_type(vector);
                if (null == this.colTypeRef) {
                    throw new SQLException("cannot initialize data chunk vector type");
                }
            } else {
                this.colTypeRef = colTypeRef;
            }
            int colTypeId = DuckDBBindings.duckdb_get_type_id(this.colTypeRef);
            this.colType = DuckDBBindings.CAPIType.capiTypeFromTypeId(colTypeId);
            if (this.colType == DuckDBBindings.CAPIType.DUCKDB_TYPE_DECIMAL) {
                int decimalInternalTypeId = DuckDBBindings.duckdb_decimal_internal_type(this.colTypeRef);
                this.decimalInternalType = DuckDBBindings.CAPIType.capiTypeFromTypeId(decimalInternalTypeId);
                this.decimalPrecision = DuckDBBindings.duckdb_decimal_width(this.colTypeRef);
                this.decimalScale = DuckDBBindings.duckdb_decimal_scale(this.colTypeRef);
            } else {
                this.decimalInternalType = DuckDBBindings.CAPIType.DUCKDB_TYPE_INVALID;
                this.decimalPrecision = -1;
                this.decimalScale = -1;
            }
            if (structFieldIdx >= 0) {
                byte[] nameUTF8 = DuckDBBindings.duckdb_struct_type_child_name(parent.colTypeRef, structFieldIdx);
                this.structFieldName = DuckDBAppender.strFromUTF8(nameUTF8);
            } else {
                this.structFieldName = null;
            }
            this.vectorRef = vector;
            this.arraySize = null == parent || parent.colType != DuckDBBindings.CAPIType.DUCKDB_TYPE_ARRAY ? 1L : DuckDBBindings.duckdb_array_type_array_size(parent.colTypeRef);
            if (this.colType.widthBytes > 0L || this.colType == DuckDBBindings.CAPIType.DUCKDB_TYPE_DECIMAL) {
                this.data = DuckDBBindings.duckdb_vector_get_data(this.vectorRef, this.vectorSize());
                if (null == this.data) {
                    throw new SQLException("cannot initialize data chunk vector data");
                }
            } else {
                this.data = null;
            }
            DuckDBBindings.duckdb_vector_ensure_validity_writable(this.vectorRef);
            this.validity = DuckDBBindings.duckdb_vector_get_validity(this.vectorRef, this.arraySize * this.parentArraySize());
            if (null == this.validity) {
                throw new SQLException("cannot initialize data chunk vector validity");
            }
            DuckDBAppender.initVecChildren(this);
        }

        void reset(long listSize) throws SQLException {
            if (null == this.parent || this.parent.colType != DuckDBBindings.CAPIType.DUCKDB_TYPE_LIST && this.parent.colType != DuckDBBindings.CAPIType.DUCKDB_TYPE_MAP) {
                throw new SQLException("invalid list column");
            }
            this.listSize = listSize;
            this.reset();
        }

        void reset() throws SQLException {
            if (null != this.data) {
                this.data = DuckDBBindings.duckdb_vector_get_data(this.vectorRef, this.vectorSize());
                if (null == this.data) {
                    throw new SQLException("cannot reset data chunk vector data");
                }
            }
            DuckDBBindings.duckdb_vector_ensure_validity_writable(this.vectorRef);
            this.validity = DuckDBBindings.duckdb_vector_get_validity(this.vectorRef, this.arraySize * this.parentArraySize());
            if (null == this.validity) {
                throw new SQLException("cannot reset data chunk vector validity");
            }
            for (Column col : this.children) {
                col.reset();
            }
        }

        void destroy() {
            for (Column cvec : this.children) {
                cvec.destroy();
            }
            this.children.clear();
            if (null != this.colTypeRef) {
                DuckDBBindings.duckdb_destroy_logical_type(this.colTypeRef);
                this.colTypeRef = null;
            }
        }

        void setNull(long vectorIdx) throws SQLException {
            if (this.colType == DuckDBBindings.CAPIType.DUCKDB_TYPE_ARRAY) {
                this.setNullOnArrayIdx(vectorIdx, 0);
                for (Column col : this.children) {
                    int i = 0;
                    while ((long)i < col.arraySize) {
                        col.setNullOnArrayIdx(vectorIdx, i);
                        ++i;
                    }
                }
            } else {
                this.setNullOnVectorIdx(vectorIdx);
                if (this.colType == DuckDBBindings.CAPIType.DUCKDB_TYPE_LIST || this.colType == DuckDBBindings.CAPIType.DUCKDB_TYPE_MAP) {
                    return;
                }
                for (Column col : this.children) {
                    col.setNull(vectorIdx);
                }
            }
        }

        void setNullOnArrayIdx(long rowIdx, int arrayIdx) {
            long vectorIdx = rowIdx * this.arraySize * this.parentArraySize() + (long)arrayIdx;
            this.setNullOnVectorIdx(vectorIdx);
        }

        void setNullOnVectorIdx(long vectorIdx) {
            long validityPos = vectorIdx / 64L;
            LongBuffer entries = this.validity.asLongBuffer();
            entries.position((int)validityPos);
            long mask = entries.get();
            long idxInEntry = vectorIdx % 64L;
            entries.position((int)validityPos);
            entries.put(mask &= 1L << (int)idxInEntry ^ 0xFFFFFFFFFFFFFFFFL);
        }

        long widthBytes() {
            if (this.colType == DuckDBBindings.CAPIType.DUCKDB_TYPE_DECIMAL) {
                return this.decimalInternalType.widthBytes;
            }
            return this.colType.widthBytes;
        }

        long parentArraySize() {
            if (null == this.parent) {
                return 1L;
            }
            return this.parent.arraySize;
        }

        long vectorSize() {
            if (null != this.parent && (this.parent.colType == DuckDBBindings.CAPIType.DUCKDB_TYPE_LIST || this.parent.colType == DuckDBBindings.CAPIType.DUCKDB_TYPE_MAP)) {
                return this.listSize * this.widthBytes();
            }
            return DuckDBBindings.duckdb_vector_size() * this.widthBytes() * this.arraySize * this.parentArraySize();
        }
    }
}

